
// AbstractProcess.cpp
// Copyright 2015 Matthew Rickard

// This file is part of dep
// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"

#include "AbstractProcess.h"
#ifdef _WIN32
#include <windows.h>
#undef min
#undef max
#endif

#ifdef unix
#include <unistd.h>
#endif
#include <string.h>
#include <time.h>
#ifdef unix
#include <sys/time.h>
#include <sys/resource.h> // required for debian 8 jessie
#include <sys/wait.h>
#endif
#include <fcntl.h>
#include <string>
#include <vector>
#include <ctype.h>
#include <errno.h>
#ifdef unix
#include <pthread.h>
#endif
#if defined _WIN32 || defined __CYGWIN__
#include <io.h>
#endif

#ifdef unix
#include <sys/stat.h> // required in debian 8 jessie? Unwanted in mingw64, redefines 'struct stat'
#endif

#include "cross_platform.h"
#include <stdlib.h>
#include <iostream>
#include "logMsg.h"
#include "MacroNames.h"
#include "Utils.h"
#include <sstream>
#include <stdlib.h>
#include <signal.h>
#ifdef __CYGWIN__
#include <sys/cygwin.h>
#endif

void Process::addArgs(std::string_view cmd) {
  //logMsg << logValue(cmd) << std::endl;
//#ifdef unix
  //logMsg << "addArgs() " << ' ' << logValue(cmd) << std::endl;
  for (const char *p = cmd.data(); *p; p++) {
    if (args.size() == 0 || isspace(*p)) {
      while (isspace(*p))
        p++;
      if (!*p)
        return;
      args.push_back("");
    }
    while (*p == '"' || *p == '\'') {
      char delimiter = *p;
      p++;
      while (*p && *p != delimiter) {
	    *(args.end() - 1) += *p;
	    p++;
      }
      if (*p)
	    p++;
    }
    if (!*p)
      break;
    if (isspace(*p)) {
      while (isspace(*p))
        p++;
      if (!*p)
        return;
      args.push_back("");
    }
    *(args.end() - 1) += *p;
  }
//#else
  //logMsg << "addArgs() " << ' ' << logValue(cmd) << std::endl;
  //addArg(cmd);
//#endif
}

Process::Process() {
  construct();
}

void Process::construct() {
  inputPipe = -1;
  outputPipe = -1;
  errorPipe = -1;
  inputFd = 0;
  outputFd = 1;
  errorFd = 2;
  running = 0;
  argv = 0;
  maxRunTime = 0;
  maxStack = 0;
  maxData = 0;
  terminateOnDestruct = 0;
}

void Process::addArg(std::string_view arg) {
  args.push_back(std::string(arg));
}

Process::Process(std::string_view cmd) {
  construct();
  addArgs(cmd.data());
}

void Process::setArgs(std::string_view cmd) {
  reset();
  addArgs(cmd.data());
}

void Process::setMaxRunTime(int maxRunTime) {
  this->maxRunTime = maxRunTime;
}

void Process::setMaxStack(size_t maxStack) {
  this->maxStack = maxStack;
}

void Process::setMaxData(size_t maxData) {
  this->maxData = maxData;
}

void Process::setError(int fd) {
  this->errorFd = fd;
}

void Process::setOutput(const std::string &filename, int flags) {
  outputFd = open(filename.c_str(), flags, 0644);
  if (outputFd == -1) {
    //int saveerrno(errno);
    exit(1);
  }
}

void Process::setError(const std::string &filename) {
  errorFd = open(filename.c_str(), O_CREAT | O_RDWR | O_TRUNC, 0644);
  if (errorFd == -1) {
    //int saveerrno(errno);
    //logMsg << logValue(saveerrno) << ' ' << logValue(ErrnoToMacroName(saveerrno)) << std::endl;
    exit(1);
  }
}

// Have to see how well Cygwin's get_osfhandle works

// Another question: the file descriptors? Windows has one mapping of handles to
// file descriptors via _open_osfhandle. Does Cygwin use that or does it have another?

void makePipe(int &readfd, int &writefd, bool inheritWrite) {
#if HANDLE_API == API_UNIX
  //showFds("makePipe started", std::cerr);
  int fildes[2];
  pipe(fildes);
  readfd  = fildes[0];
  writefd = fildes[1];
  // Set one of the FDs to close on exec (you can set both, because one of them
  // will be dup()ed to an FD that stays open on exec).
  fcntl(readfd, F_SETFD, FD_CLOEXEC);
  fcntl(writefd, F_SETFD, FD_CLOEXEC);
  //std::cerr << "makePipe() made " << logValue(readfd)
            //<< " and " << logValue(writefd) << std::endl;
#else
  HANDLE readHandle, writeHandle;
  CreatePipe(&readHandle, &writeHandle, 0, 0);
  // Set one of the file handles to stay open on exec
  if (inheritWrite)
    DuplicateHandle(GetCurrentProcess(), writeHandle, GetCurrentProcess(), &writeHandle, 0, TRUE, DUPLICATE_CLOSE_SOURCE | DUPLICATE_SAME_ACCESS);
  else
    DuplicateHandle(GetCurrentProcess(), readHandle, GetCurrentProcess(), &readHandle, 0, TRUE, DUPLICATE_CLOSE_SOURCE | DUPLICATE_SAME_ACCESS);
  // And convert the handles to FDs.
  readfd = _open_osfhandle((long long)readHandle, O_RDONLY);
  writefd = _open_osfhandle((long long)writeHandle, 0);
#endif
}

void Process::pipeInput() {
  makePipe(inputFd, inputPipe, 0);
}

void Process::pipeOutput() {
  makePipe(outputPipe, outputFd, 1);
}

void Process::pipeError() {
  makePipe(errorPipe, errorFd, 1);
}

int Process::run() {
  if (!start()) {
    return -1;
  }
  return waitForExit();
}

static bool executable1(const char *filename, std::string &filename2) {
#if HANDLE_API == API_UNIX
  struct stat status;
  bool result;
  if (stat(filename, &status))
    result = false;
  else if (geteuid() == 1)
    result = status.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH);
  else if (status.st_uid == geteuid())
    result = status.st_mode & S_IXUSR;
  else if (status.st_gid == getegid())
    result = status.st_mode & S_IXGRP;
  else
    result = status.st_mode & S_IXOTH;
  //logMsg << logValue(filename) << ": " << logValue(result) << std::endl;
  return result;
#else
  filename2 = filename;
  std::string endbit(RECAP(filename2.substr(filename2.size() - 4)));
  if (endbit != ".EXE") {
    filename2 += ".EXE";
    filename = filename2.c_str();
  }
  //logMsg << logValue(filename) << ' ' << logValue(endbit) << std::endl;
  DWORD dontcare;
  bool result = GetBinaryType(filename, &dontcare);
  int error(GetLastError());
  if (error == ERROR_FILE_NOT_FOUND && !result) {
    //logMsg << logValue(filename) << ' ' << logValue(result) << ' ' << logValue(error) << ' ' << WinErrorToMacroName(error) << std::endl;
    return 0;
  }
  if (error == ERROR_PATH_NOT_FOUND && !result) {
    //logMsg << logValue(filename) << ' ' << logValue(result) << ' ' << logValue(error) << ' ' << WinErrorToMacroName(error) << std::endl;
    return 0;
  }
  if (/*error == ERROR_ALREADY_EXISTS &&*/ result) {
    //logMsg << logValue(filename) << ' ' << logValue(result) /*<< ' ' << logValue(error) << ' ' << WinErrorToMacroName(error)*/ << std::endl;
    return 1;
  }
  logMsg << logValue(filename) << ' ' << logValue(result) << ' ' << logValue(error) << ' ' << WinErrorToMacroName(error) << std::endl;
  lineabort();
  //return false;
#endif
}

static bool executable2(std::string subPath, const char *filename, std::string &fullname) {
  if (subPath.length() == 0)
    return executable1(filename, fullname);
  subPath.append("/");
  subPath.append(filename);
  return executable1(subPath.c_str(), fullname);
}

#ifdef unix
#define PATH_SEPARATOR ':'
#elif defined _WIN32
#define PATH_SEPARATOR ';'
#endif

static bool searchPath(const char *filename,
    const char *path, std::string &fullname) {
  //logMsg << logValue(filename) << std::endl;
  if (strchr(filename, '/'))
    return executable1(filename, fullname);
  const char *p1 = path;
  const char *p2 = strchr(path, PATH_SEPARATOR);
  while (p2) {
    //logMsg << logValue(p2) << std::endl;
    std::string subPath(path, p1 - path, p2 - p1);
    if (executable2(subPath, filename, fullname))
      return true;
    p1 = p2 + 1;
    p2 = strchr(p1, PATH_SEPARATOR);
  }
  //logMsg << logValue(p2) << std::endl;
  if (executable2(p1, filename, fullname))
    return true;
  return false;
}

void logclose(int fd) {
  //logMsg << "closing " << logValue(fd) << std::endl;
  close(fd);
}

void showFds(const std::string &prefix, std::ostream &os) {
#if defined __MINGW64__ || defined __MINGW32__
  //bork bork bork
  lineabort();
#else
  std::ostringstream oss; // send them out all at once. This is coming from two processes
  for (int i = 0; i < 50; i++) {
    int result = fcntl(i, F_GETFD);
    if (result != -1)
      oss << prefix << ' ' << i << ' ' << result << '\n';
  }
  os << oss.str() << std::flush;
#endif
}

void Process::showFds(std::ostream &os) {
  std::ostringstream oss;
  oss << getArgs() << ' ' << logValue(child);
  ::showFds(oss.str(), os);
}

void Process::setDir(const char *argdir) {
  dir = argdir;
}

int Process::start() {
  if (getenv("SHOWPROCESS")) {
    logMsg << "Process::start() running ";
    printArgs(std::cerr);
    std::cerr << std::endl;
  }
  std::string fullname;
  std::string program(getProgram());
  //logMsg << logValue(program) << std::endl;
  if (hasArgs()) {
    if (!searchPath(program.c_str(), getenv("PATH"), fullname)) {
      // Use stdio instead of iostream because backTrace can call this
      // during pre-main() initialization and if plextestbacktrace isn't
      // available, well, succinct diagnostic beats gratuitous coredump.
      fprintf(stderr, "Process::start(): couldn't find %s in PATH\n", program.c_str());
#if PROCESS_API == API_UNIX
      child = 0;
      // this stops pipelines from hanging
      if (inputPipe  != -1) logclose(inputFd);
      if (outputPipe != -1) logclose(outputFd);
      if (errorPipe  != -1) logclose(errorFd);
      return false;
#else
      // toodo: do the Windows equivalent here
#endif
    }
  }
#if PROCESS_API == API_UNIX
  int retrySeconds = 1;
  retry:
  if (child = fork()) {
    if (child == -1) {
      int error = errno;
      logMsg << "fork() failed. " << logValue(child) << ' ' << logValue(error)
        << ' ' << ErrnoToMacroName(error) << std::endl;
      if (isatty(0)) {
        //std::cout << "Press Enter to Retry or CtrlC to end" << std::endl;
        //std::string s;
        //getline(std::cin, s);
        std::cout << "Retrying in " << retrySeconds << "s or press CtrlC to end" << std::endl;
        sleep(retrySeconds);
        retrySeconds++;
        goto retry;
      }
      exit(1);
    }
    if (getenv("SHOWPROCESS"))
      logMsg << logValue(child) << std::endl;
    //printArgs(std::cerr);
    //std::cerr << std::endl;
    //showFds(std::cerr);
    if (inputPipe  != -1) logclose(inputFd);
    if (outputPipe != -1) logclose(outputFd);
    if (errorPipe  != -1) logclose(errorFd);
  }
  else {
    exec();
  }
  gettimeofday(&startTime, 0);
#else
  //logMsg << logValue(getArgs()) << std::endl;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  ZeroMemory(&si, sizeof si);
  si.cb = sizeof si;
  si.dwFlags = STARTF_USESTDHANDLES;
  si.hStdInput = (HANDLE)_get_osfhandle(inputFd);
  si.hStdOutput = (HANDLE)_get_osfhandle(outputFd);
  si.hStdError = (HANDLE)_get_osfhandle(errorFd);
  //logMsg << logValue(inputFd) << std::endl;
  //logMsg << logValue(outputFd) << std::endl;
  //logMsg << logValue(errorFd) << std::endl;
  //logMsg << logValue(si.hStdInput) << std::endl;
  //logMsg << logValue(si.hStdOutput) << std::endl;
  //logMsg << logValue(si.hStdError) << std::endl;
  GetSystemTimeAsFileTime((FILETIME *)&startTime);
  BOOL result = CreateProcess(
    fullname.c_str(),

    (char *)getArgs().c_str(),
    0, 0, // process security attributes, thread security attributes
    1, // inherit handles
    0, // priority class (0 = lowest of parent's class and normal)
    0, // environment (0 = same as calling process)
    0, // working directory (0 = same as calling process)
    &si,
    &pi);

  int error = 0;
  if (!result)
    error = GetLastError();

  child = (pid_t)pi.hProcess;
  CloseHandle(pi.hThread);
  if (inputPipe  != -1) {
    close(inputFd);
  }
  if (outputPipe != -1) {
    close(outputFd);
  }
  if (errorPipe  != -1) {
    close(errorFd);
  }

  if (!result) {
    logMsg << logValue(result) << ' ' << logValue(error) << ' ' << WinErrorToMacroName(error) << std::endl;
    logMsg << "Process::start() aborting because of error" << std::endl;
    lineabort();
  }

  //logMsg << "Process::start() pretty much done" << std::endl;
#endif
  running = 1;
  return true;
}

void Process::exec() {
#if PROCESS_API == API_UNIX
  if (getenv("SHOWPROCESS"))
    logMsg << logValue(child) << std::endl;
  //showFds(std::cerr);
  saveerrfd = 2;
  if (inputFd    !=  0) dup2(inputFd,  0);
  if (outputFd   !=  1) dup2(outputFd, 1);
  if (errorFd    !=  2) {
    saveerrfd = dup(2); // copy used for outputting errors if execvp() fails
    dup2(errorFd, 2);
    fcntl(saveerrfd, F_SETFD, FD_CLOEXEC);
  }
  if (inputFd  > 2) close(inputFd);
  if (outputFd > 2) close(outputFd);
  if (errorFd  > 2) close(errorFd);
  if (inputPipe  != -1) close(inputPipe);
  if (outputPipe != -1) close(outputPipe);
  if (errorPipe  != -1) close(errorPipe);
  if (maxRunTime > 0)   alarm(maxRunTime);
  struct rlimit rl;
  if (maxStack > 0) {
    rl.rlim_cur = rl.rlim_max = maxStack;
    setrlimit(RLIMIT_STACK, &rl);
  }
  if (maxData > 0) {
    rl.rlim_cur = rl.rlim_max = maxData;
    setrlimit(RLIMIT_DATA, &rl);
  }

  for (std::map<std::string, std::string>::iterator i = envmap.begin(); i != envmap.end(); i++) {
    //logMsg << "setting " << i->first << " to " << i->second << std::endl;
    ::setenv(i->first.c_str(), i->second.c_str(), 1);
  }
  if (dir.size())
    chdir(dir.c_str());

  if (getenv("SHOWPROCESS"))
    logMsg << "calling execvp" << std::endl;
  execvp(getProgram().c_str(), const_cast<char *const *>(getArgv()));
  int errno1 = errno;
  if (getenv("SHOWPROCESS"))
    logMsg << "execvp failed " << logValue(errno1) << std::endl;
  FILE *saveerr = fdopen(saveerrfd, "w");
  fprintf(saveerr, "Could not exec \'%s\', error = %d %s\n",
    getProgram().c_str(), errno1, ErrnoToMacroName(errno1));
  fflush(saveerr);
#endif
  _exit(1);
}

void Process::safeclose(int fd) {
  //logMsg << "Process::safeclose closing fd " << fd << std::endl;
  if (fd != -1) {
    int result = close(fd);
    if (result) {
      int someerrno(errno);
      logMsg << "close() " << logValue(result) << ' ' << ".\n";
      logMsg << "Couldn't close file descriptor " << fd << ". "
        << logValue(someerrno) << ' ' << ErrnoToMacroName(someerrno) << ".\n";
      logMsg << "Args were " << getArgs() << '\n';
      logMsg << "Usually this is because an stdio_filebuf was attached to it\n";
      logMsg << "and closed the file automatically.\n";
      logMsg << "Drop the fd from the process like this:\n";
      logMsg << "  p.pipeOutput();\n";
      logMsg << "  p.start();\n";
      logMsg << "  File f(\"(output stream)\", p.outputPipe);\n";
      logMsg << "  p.outputPipe = -1;\n";
      logMsg << "Aborting.\n";
      lineabort();
    }
  }
}

void Process::reset() {
  args.clear();
  delete[] argv;
  argv = 0;
}

Process::~Process() {
  //logMsg << "Destroying a Process " << logValue(this) << std::endl;
  // Use a version of close that has loud, easy-to-trace error messages if something goes
  // wrong. Microsoft CRT in particular produces quiet, hard-to-trace, wild-goose-chase error
  // messages.
  closeProcess();
  if (running && terminateOnDestruct) {
#if PROCESS_API == API_UNIX
    logMsg << "Terminating process " << logValue(argv[0]) << ' ' << logValue(child) << std::endl;
#ifdef __CYGWIN__
    int winpid = cygwin_internal(CW_CYGWIN_PID_TO_WINPID, child);
    HANDLE h = OpenProcess(PROCESS_TERMINATE, 0, winpid);
    int result = TerminateProcess(h, 19);
    if (!result) {
      int error = GetLastError();
      logMsg << logValue(error) << ' ' << WinErrorToMacroName(error)
        << ' ' << logValue(winpid) << ' ' << logValue(h) << std::endl;
    }
    CloseHandle(h);
#else
    int result = kill(child, SIGKILL);
    if (result) {
      int error = errno;
      logMsg << logValue(error) << ' ' << ErrnoToMacroName(error) << std::endl;
    }
#endif
#else
    lineabort();
#endif
  }
  delete[] argv;
}

void Process::closeProcess() {
  safeclose(inputPipe);
  safeclose(outputPipe);
  safeclose(errorPipe);
}

void Process::printArgs(std::ostream &os) {
  const char **argv(getArgv());
  for (const char **arg = argv; *arg; arg++) {
    if (arg != argv)
      os << ' ';
    os << *arg;
  }
}

const char **Process::getArgv() {
  if (!argv) {
    argv = new const char *[args.size() + 1];
    int argc = 0;
    for (std::vector<std::string>::const_iterator i = args.begin(); i < args.end(); i++) {
      argv[argc++] = i->c_str();
    }
    argv[argc] = 0;
  }
  return argv;
}

int Process::hasArgs() {
  return args.size();
}

std::string Process::getProgram() {
  const char *argv0 = getArgv()[0];
  if (!argv0) {
    logMsg << "No program!" << std::endl;
    return "";
  }
  std::string result = argv0;
  //logMsg << logValue(result) << std::endl;
  size_t space = result.find(' ');
  if (space != std::string::npos) {
    result.erase(space);
  }
  //logMsg << logValue(result) << std::endl;
  return result;
}

void Process::processWaitResult(int argWaitResult) {
  this->waitResult = argWaitResult;
  gettimeofday(&finishTime, 0);
  running = 0;
  runTime = (finishTime.tv_sec - startTime.tv_sec);
  if (finishTime.tv_usec > startTime.tv_usec)
    runTime++;
}

int Process::waitForExit() {
  terminateOnDestruct = 0;
  if (inputPipe != -1) {
    close(inputPipe); // need to to this when using waitForExit
    inputPipe = -1; // and probably this
  }
#if PROCESS_API == API_UNIX
  if (getenv("SHOWPROCESS"))
    logMsg << "Process::waitForExit calling waitpid" << std::endl;
  pid_t result = waitpid(child, &waitResult, 0);
  if (getenv("SHOWPROCESS"))
    logMsg << "Process::waitForExit called waitpid " << logValue(waitResult)
      << ' ' << waitString(waitResult) << std::endl;
  gettimeofday(&finishTime, 0);
  running = 0;
  runTime = (finishTime.tv_sec - startTime.tv_sec);
  if (finishTime.tv_usec > startTime.tv_usec)
    runTime++;
  if (result == -1)
    return -1;
  //logMsg << logValue(waitResult) << std::endl;
  return waitResult;
#else
  DWORD result = WaitForSingleObject((HANDLE)child, INFINITE);
  GetSystemTimeAsFileTime((FILETIME *)&finishTime);
  running = 0;

#if defined __MINGW32__ && !defined __MINGW64__
  runTime = (finishTime - startTime) / 10000000;
#else
  runTime = (finishTime.tv_sec - startTime.tv_sec);
  if (finishTime.tv_usec > startTime.tv_usec)
    runTime++;
#endif
  if (int(result) == -1)
    return -1;

  if (result != WAIT_OBJECT_0) {
    logMsg << "That's not what I was expecting. " << logValue(result) << std::endl;
  }
  DWORD status;
  BOOL result2 = GetExitCodeProcess((HANDLE)child, &status);
  if (!result2) {
    logMsg << "That's not what I was expecting. " << logValue(result2) << std::endl;
  }
  CloseHandle((HANDLE)child);
  return status;
#endif
}

int Process::getRunTime() {
  return runTime;
}

std::string Process::getArgs() {
  std::string result;
  const char **argv = getArgv();
  while (*argv) {
    if (result.size())
      result.push_back(' ');
    result.append(*argv);
    argv++;
  }
  return result;
}

void Process::setenv(const char *env, const char *val) {
  envmap[env] = val;
}

void Process::pipeTo(Process &b) {
  pipeOutput();
  b.inputFd = outputPipe;
}

Pipeline::Pipeline() { }
Pipeline::Pipeline(const std::string &cmd) {
  addProcess(cmd);
}

void Pipeline::addProcess(const std::string &cmd) {
  processVector.push_back(new Process);
  //logMsg << logValue(cmd) << " at " << processVector.back() << std::endl;
  processVector.back()->addArgs(cmd);
  if (processVector.size() >= 2) {
    processVector[processVector.size() - 2]->pipeTo(*processVector.back());
  }
}

void Pipeline::setOutput(const char *s) {
  processVector.back()->setOutput(s);
}

int Pipeline::start() {
  for (auto i = processVector.begin(); i != processVector.end(); i++) {
    (*i)->envmap = envmap;
    int result = (*i)->start();
    //logMsg << logValue(result) << std::endl;
    if (!result)
      return 0;
  }
  return 1;
}

int Pipeline::run() {
  start();
  return waitForExit();
}

int Pipeline::waitForExit() {
  int result = 0;
  for (auto i = processVector.begin(); i != processVector.end(); i++)
    if (result)
      (*i)->waitForExit();
    else
      result = (*i)->waitForExit();
  return result;
}

void Pipeline::clear() {
  for (auto i = processVector.begin(); i != processVector.end(); i++)
    delete *i;
  processVector.clear();
}

Pipeline::~Pipeline() {
  for (auto i = processVector.begin(); i != processVector.end(); i++)
    delete *i;
}

const char *findUnquoted(const char *s, const char *lookingFor) {
  std::string longerString = std::string(lookingFor) + "'\"\\";
  for (;;) {
    const char *s2 = strpbrk(s, longerString.c_str());
    if (!s2)
      return 0;
    if (*s2 == 0)
      return s2;
    if (strchr(lookingFor, *s2))
      return s2;
    if (*s2 == '\\') {
      s += 2;
      continue;
    }
    if (*s2 == '\'') {
      s2 = findUnquoted(s2 + 1, "\'");
      if (!s2)
        return 0;
      if (*s2) {
        s = s2 + 1;
        continue;
      }
      return s2;
    }
    if (*s2 == '\"') {
      s2 = findUnquoted(s2 + 1, "\"");
      if (!s2)
        return 0;
      if (*s2) {
        s = s2 + 1;
        continue;
      }
      return s2;
    }
    logMsg << logValue(s) << std::endl;
    logMsg << logValue(s2) << std::endl;
    lineabort();
  }
}

int findUnquotedPos(const char *s, const char *lookingFor) {
  const char *result = findUnquoted(s, lookingFor);
  if (result)
    return result - s;
  return -1;
}

int System(const std::string &cmd, int outFd, int errorFd,
    const char *inputStr, int flags, Pipeline *pipeline) {
  Pipeline pl2;
  if (!pipeline)
    pipeline = &pl2;
  size_t bar1 = 0;
  int closeOut = 0;
  //logMsg << logValue(cmd) << std::endl;
  for (;;) {
    int barx = findUnquotedPos(cmd.c_str() + bar1, "|>");
    //logMsg << logValue(cmd) << std::endl;
    //logMsg << logValue(bar1) << std::endl;
    //logMsg << logValue(barx) << std::endl;

    if (barx >= 0) {
      barx += bar1;
      //logMsg << logValue(barx) << std::endl;
      //logMsg << "Checkpoint" << std::endl;
      if (cmd[barx] == '>') {
        pipeline->addProcess(stripWhitespace(cmd.substr(bar1, barx - bar1)));
        std::string outName = stripLeadingWhitespace(cmd.substr(barx + 1));
        outFd = open(outName.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0666);
        if (outFd < 0) {
          int error = errno;
          logMsg << logValue(error) << ' ' << ErrnoToMacroName(error) << std::endl;
          errno = error;
          return 13;
        }
        closeOut = 1;
        break;
      }
      else if (cmd[barx] == '|') {
        pipeline->addProcess(cmd.substr(bar1, barx - bar1 - 1));
        bar1 = barx + 1;
      }
      else {
        logMsg << logValue(barx) << std::endl;
        lineabort();
      }
      bar1 = barx + 1;
      //logMsg << "Checkpoint" << std::endl;
    }
    else {
      //logMsg << "Checkpoint" << std::endl;
      pipeline->addProcess(cmd.substr(bar1));
      break;
    }
  }
  if (outFd > -1) {
    pipeline->processVector.back()->outputFd = outFd;
  }
  if (errorFd > -1) {
    pipeline->processVector.back()->errorFd = errorFd;
  }
  if (inputStr) {
    pipeline->processVector.front()->pipeInput();
    // toodo: this can only do very short strings. Improve
    write(pipeline->processVector.front()->inputPipe, inputStr, strlen(inputStr));
  }
  //int result = pipeline.run();
  //logMsg << "starting pipeline" << std::endl;
  int result1 = pipeline->start();
  //logMsg << "started pipeline, " << logValue(result1) << std::endl;
  if (!result1)
    return 65535;
  if (flags & SYSTEMFLAG_ASYNC) {
    //logMsg << "Returning now" << std::endl;
    return 0;
  }
  //logMsg << "Waiting for exit" << std::endl;
  int result = pipeline->waitForExit();
  if (closeOut)
    close(outFd);
  return result;
}

// php-cgi can have the script and arguments on the command line, or as the environment
// variables SCRIPT_FILENAME and QUERY_STRING. It is slightly better if they are
// passed in as environment variables, because the script becomes an unused _GET
// variable otherwise. If you want to post stuff in, they have to be environment variables
// and there has to be a bunch more enviro variables.

int phpCgi(const char *script, const char *args, const char *input, const File &argout) {
  static int setup = 0;
  if (!setup) {
    setenv("CONTENT_TYPE", "application/x-www-form-urlencoded", 1);
    setenv("REQUEST_METHOD", "POST", 1);
    setenv("REDIRECT_STATUS", "true", 1);
    setup = 1;
  }
  setenv("SCRIPT_FILENAME", script, 1);
  setenv("QUERY_STRING", args, 1);
  if (input) {
    //char buf[16];
    char buf[32];
    sprintf(buf, "%ld", (long)strlen(input));
    setenv("CONTENT_LENGTH", buf, 1);
  }
  else
    unsetenv("CONTENT_LENGTH"); // needed for ListFormNewTest
  File &out((File &)argout); // break the const
  //logMsg << logValue(out.getFd(O_CREAT | O_TRUNC | O_RDWR)) << std::endl; // A quick test to ensure these aren't leaking
  int result = System("php-cgi | sed '/X-Powered-By:/d;/Set-Cookie: PHPSESSID/d;"
    "/Expires:/d;/Cache-Control/d;/exiting now because of ONERROR_EXIT/d' | xmlbeautify",
    out.getFd(O_CREAT | O_TRUNC | O_RDWR), -1, input);
  // TOODO: Do this: Process("php-cgi") < input | xmlbeautify > out;
  return result;
}

std::string waitString(int status) {
  std::ostringstream os;
  if (status == -1)
    os << logValue(status) << " not a valid exit status";
  else if (WIFEXITED(status))
    os << "exitcode " << WEXITSTATUS(status);
  else if (WIFSIGNALED(status)) {
    os << logValue(status) << " terminated by signal " << WTERMSIG(status)
      << ' ' << SignalToMacroName(WTERMSIG(status));
#ifndef __MINGW32__
    if (WCOREDUMP(status))
      os << " (core dumped)";
#endif
  }
#ifndef __MINGW32__
  else if (WIFSTOPPED(status))
    os << logValue(status) << " stopped by signal " << WSTOPSIG(status);
  else if (WIFCONTINUED(status))
    os << logValue(status) << " stopped by SIGCONT";
#endif
  else
    os << "unknown process status " << status << std::endl;
  return os.str();
}

