
// AbstractProcess.h
// Copyright 2008 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

// (This is AbstractProcess.h not Process.h to distinguish
// it from /usr/include/process.h under cygwin/mingw)

#ifndef ABSTRACT_PROCESS_H
#define ABSTRACT_PROCESS_H

#define API_UNIX    1
#define API_WINDOWS 2

#ifdef __CYGWIN__
#define PROCESS_API API_UNIX // would like this to be API_WINDOWS. API_UNIX works tho
#define HANDLE_API API_UNIX
#elif defined __unix__
#define PROCESS_API API_UNIX
#define HANDLE_API API_UNIX
#elif defined _WIN32
#define PROCESS_API API_WINDOWS
#define HANDLE_API API_WINDOWS
#endif

#define SYSTEMFLAG_ASYNC 1

#include <stdio.h>
#include <vector>
#include <map>
#include <string>
//#ifdef unix
#if defined _WIN32 && (!defined __CYGWIN__ && !defined __MINGW64__)
#else
#include <sys/time.h>
#endif
#include <sys/types.h>

#if defined _WIN32 || defined __CYGWIN__
#include <windows.h>
#endif

#ifdef _WIN32
#include <basetsd.h> // added 19/2/2015 for UINT64 for mingw32
#endif

#include <fcntl.h>
#include "File.h"

struct Process {
  Process();
  Process(std::string_view cmd);
  void setMaxRunTime(int maxRunTime);
  void setMaxStack(size_t maxStack);
  void setMaxData(size_t maxData);
  void setError(int fd);
  void setOutput(const std::string &filename, int flags = O_CREAT | O_WRONLY | O_TRUNC);
  void setError(const std::string &filename);
  void setDir(const char *dir);
  void pipeInput();
  void pipeOutput();
  void pipeTo(Process &b);
  void pipeError();
  int run();
  void printArgs(std::ostream &);
  int start();
  int waitForExit();
  void processWaitResult(int waitResult);
  const char **getArgv();
  int hasArgs();
  std::string getProgram();
  std::string getArgs();
  int getRunTime();
  void closeProcess();
  ~Process();
  void addArgs(std::string_view args);
  void setArgs(std::string_view args);
  void addArg(std::string_view arg);
  void setenv(const char *env, const char *val);
  void reset();
  void safeclose(int fd);
  void exec() __attribute__((noreturn));

  int inputPipe, outputPipe, errorPipe;
  int inputFd, outputFd, errorFd;
  pid_t child;
  int running;
  std::vector<std::string> args;
  const char **argv;
#if PROCESS_API == API_UNIX
  int waitResult;
#else
  unsigned long waitResult;
#endif
  std::map<std::string, std::string> envmap;
  int terminateOnDestruct;
  std::string dir;

private:
  void construct();
  void showFds(std::ostream &os);

  int maxRunTime;
  size_t maxStack;
  size_t maxData;
  int runTime;
#if defined _WIN32 && (!defined __CYGWIN__ && !defined __MINGW64__)
  UINT64 startTime, finishTime;
#else
  struct timeval startTime, finishTime;
#endif
  int saveerrfd;
};

void showFds(const std::string &prefix, std::ostream &os);

struct Pipeline {
  Pipeline();
  Pipeline(const std::string &cmd);
  void addProcess(const std::string &cmd);
  void setOutput(const char *s);
  int run();
  int start();
  int waitForExit();
  ~Pipeline();
  void clear();

  std::vector<Process *> processVector;
  std::map<std::string, std::string> envmap;
};

// Usually from C/C++ you want to use Process or Pipeline.
// System is for processing longhand command lines obtained from somewhere else.
// Don't build up a long command line just to have System tear it back down, and then use
// construct a Pipeline when you really just needed a Process.

int System(const std::string &cmd, int outFd = -1, int errorFd = -1, const char *inputStr = 0,
  int flags = 0, Pipeline *pipeline = 0);

int phpCgi(const char *script, const char *args, const char *input, const File &argout);

std::string waitString(int status);

const char *findUnquoted(const char *s, const char *lookingFor);

#endif

