
// File.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "File.h"
#include "logMsg.h"
#include "cross_platform.h"
#include <unistd.h>
#include <fcntl.h>
#include <malloc.h>
#if defined _WIN32 && !defined __CYGWIN__
#include <windows.h>
#endif
#include "MacroNames.h"

#if !defined __MINGW32__
#include <sys/mman.h> // not available in Ming
#endif

#include <stdlib.h>
#include <sys/stat.h>

const int POINTS_TO_NOTHING  = 0;
const int POINTS_TO_IOSTREAM = 1;
const int POINTS_TO_FD       = 2;
const int POINTS_TO_OSTREAM  = 3;

std::iostream *File::getIOStream(std::ios::openmode flags) {

  //logMsg << logValue(getHandleCount()) << std::endl;

  int openflags = 0;
  //if (flags & (std::ios::in | std::ios::out))
    //openflags = O_RDWR | O_CREAT;
  if ((flags & std::ios::in) && (flags & std::ios::out))
    openflags = O_RDWR | O_CREAT;
  else if (flags & std::ios::out)
    openflags = O_WRONLY | O_CREAT;
  else // if (flags & std::ios::in)
    openflags = O_RDONLY;
  if (flags & std::ios::trunc)
    openflags |= O_TRUNC;
  //if (flags & std::ios::binary)
    //logMsg << "That's binary" << std::endl;
  //if (flags & std::ios::floatfield)
    //logMsg << "That's floatfield" << std::endl;
  //lineabort();

  //logMsg << logValue(getHandleCount()) << std::endl;
  getFd(openflags);
  //logMsg << logValue(getHandleCount()) << std::endl;

  if ((pointsTo == POINTS_TO_NOTHING /*|| pointsTo == POINTS_TO_FD*/) && filename.size()) {
    //logMsg << "checkpoint " << logValue(filename) << ' ' << logValue(flags) << std::endl;
    std::fstream *fs = new std::fstream(filename.c_str(), flags);
    ptr = fs;
    if (!fs->good()) {
      //logMsg << "Error: Cannot open " << logValue(filename) << ". Exiting" << std::endl;
      //exit(1);
      // Exiting is a little harsh. It precludes replacing the file
      logMsg << "Error: Cannot open " << logValue(filename) << std::endl;
      //exit(1);
    }
    pointsTo = POINTS_TO_IOSTREAM;
  }
  else if (pointsTo == POINTS_TO_FD && fd >= 0) {
#if defined __MINGW64__ || defined __MINGW32__
    lineabort(); // When tomorrow comes
#else
    //logMsg << logValue(getHandleCount()) << std::endl;
    buf = new __gnu_cxx::stdio_filebuf<char>(fd, flags, BUFSIZ);
    //logMsg << logValue(getHandleCount()) << std::endl;
    if (!buf->is_open()) {
      std::cerr << __FILE__ << ' ' << __LINE__ <<
        " File::getIOStream can't upgrade an fd. often this is due to mode errors\n";
    }
    //logMsg << logValue(getHandleCount()) << std::endl;
    std::iostream *streamptr = new std::iostream(buf);
    //logMsg << logValue(getHandleCount()) << std::endl;
    // fd?
    ptr = streamptr;
    pointsTo = POINTS_TO_IOSTREAM;
#endif
  }
  if (pointsTo == POINTS_TO_IOSTREAM) {
    //logMsg << logValue(getHandleCount()) << std::endl;

    return (std::iostream *)ptr;
  }
  else if (pointsTo == POINTS_TO_OSTREAM) {
    logMsg << "This points to an ostream not an iostream" << std::endl;
    lineabort();
  }
  else
    lineabort();
}

std::ostream *File::getOStream(std::ios::openmode flags) {
  if (pointsTo == POINTS_TO_OSTREAM)
    return (std::ostream *)ptr;
  return getIOStream(flags);
}

std::istream *File::getIStream(std::ios::openmode flags) {
  // toodo: this is a very feeble implementation that can't handle actual ostreams
  return getIOStream(flags);
}

File::File(const std::string &filename): filename(filename), ptr(0), pointsTo(0), fd(-1), buf(0) {
  dontDelete = 0;
}

File::File(const char *filename): filename(filename), ptr(0), pointsTo(0), fd(-1), buf(0) {
  dontDelete = 0;
}

File::File(): ptr(0), pointsTo(0), fd(-1), buf(0) {
  dontDelete = 0;
}

File::File(const std::string &filename, long fd):
    filename(filename), ptr((void *)(long long)fd), pointsTo(POINTS_TO_FD), fd(fd), buf(0) {
  dontDelete = 0;
}

File::File(const std::string &name, long fd, std::ostream &os): filename(name),
    ptr(&os), pointsTo(POINTS_TO_OSTREAM), fd(fd), buf(0) {
  dontDelete = 1;
  //logMsg << "There's that ostream constructor" << std::endl;
  //mapping = 0;
}

void File::close() {
  //logMsg << "Closing " << filename << ' ' << logValue(dontDelete) << std::endl;
  if (dontDelete)
    return;
  if (pointsTo == POINTS_TO_IOSTREAM) {
    ((std::iostream *)ptr)->flush();
    //logMsg << "Deleting ptr for " << filename << std::endl;
    delete (std::iostream *)ptr;
    if (fd >= 0)
      ::close(fd);
  }
  else if (pointsTo == POINTS_TO_FD)
    ::close(fd);
  else if (pointsTo)
    lineabort();
  ptr = 0;
  fd = -1;
  pointsTo = 0;
  if (buf) {
    delete buf;
    buf = 0;
  }
}

File::~File() {
  close();
  //if (mapping)
    //munmap((void *)mapping, mapSize);
}

int File::getFd(int flags) {
  if (!pointsTo) {
    //logMsg << "Opening " << filename << " as fd" << std::endl;
    fd = open(filename.c_str(), flags, 0666);
    if (fd < 0) {
      int error = errno;
      if (errno != EEXIST)
        logMsg << "Could not open " << logValue(filename) << ' ' << logValue(fd)
          << ' ' << logValue(errno) << ' ' << ErrnoToMacroName(errno) << std::endl;
      return -1;
    }
    pointsTo = POINTS_TO_FD;
    return fd;
  }
  //logMsg << logValue(pointsTo) << ' ' << logValue(POINTS_TO_IOSTREAM) << std::endl;
  if (pointsTo == POINTS_TO_IOSTREAM && fd < 0) {
    std::iostream *ios = (std::iostream *)ptr;
    logMsg << "Can't downgrade from a iostream to a fd unfortunately. " << logValue(fd) << std::endl;
    std::streambuf *buf = ios->rdbuf();
    logMsg << logValue((long long)buf) << std::endl;
    std::filebuf *fbuf = dynamic_cast<std::basic_filebuf<char> *>(buf);
    logMsg << logValue((long long)fbuf) << std::endl;
    //lineabort();
  }
  return fd;
}

int File::getTmpFd() {
  if (!pointsTo) {
    //logMsg << "Opening " << filename << " as fd" << std::endl;
    fd = mkstemp(filename.data());
    if (fd < 0) {
      int error = errno;
      if (errno != EEXIST)
        logMsg << "Could not open " << logValue(filename) << ' ' << logValue(fd)
          << ' ' << logValue(errno) << ' ' << ErrnoToMacroName(errno) << std::endl;
      return -1;
    }
    pointsTo = POINTS_TO_FD;
    return fd;
  }
  //logMsg << logValue(pointsTo) << ' ' << logValue(POINTS_TO_IOSTREAM) << std::endl;
  if (pointsTo == POINTS_TO_IOSTREAM && fd < 0) {
    std::iostream *ios = (std::iostream *)ptr;
    logMsg << "Can't downgrade from a iostream to a fd unfortunately. " << logValue(fd) << std::endl;
    std::streambuf *buf = ios->rdbuf();
    logMsg << logValue((long long)buf) << std::endl;
    std::filebuf *fbuf = dynamic_cast<std::basic_filebuf<char> *>(buf);
    logMsg << logValue((long long)fbuf) << std::endl;
    //lineabort();
  }
  return fd;
}

int getHandleCount() {

#if defined _WIN32

#if defined __MINGW64__
  unsigned long handles;
#else
  unsigned handles;
#endif
  GetProcessHandleCount(GetCurrentProcess(), &handles);
  return handles;

#else

  lineabort();

#endif

}

FileMap::FileMap(const char *name) {
  mapFile(name);
}

FileMap::~FileMap() {
#if defined __MINGW32__
  lineabort();
#else
  if (mapping)
    munmap((void *)mapping, mapSize);
#endif
}

const char *FileMap::mapFile(const char *filename) {
#if defined __MINGW32__
  lineabort();
#else
  struct stat statbuf;
  int fd = open(filename, O_RDONLY);
  fstat(fd, &statbuf);
  mapSize = statbuf.st_size + 1;
  mapping = (const char *)mmap(0, mapSize, PROT_READ, MAP_PRIVATE, fd, 0);
  close(fd);
  return mapping;
#endif
};

