
// ParseTree.cpp
// Copyright 2015 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "ParseTree.h"
#include <iostream>
#include "cross_platform.h"
#include "Utils.h"
#include <unistd.h>
#include "logMsg.h"
#include "quote.h"
#include <sstream>
#include <string.h>
#include <set>

ParseTree::~ParseTree() {
  delete trunk;
}

BranchNode::~BranchNode() {
  for (std::vector<TreeNode *>::iterator i = v.begin(); i != v.end(); i++) {
    delete *i;
  }
}

TreeNode::~TreeNode() {
}

LeafNode::~LeafNode() {
}

ParseTree::ParseTree(YY_BUFFER_STATE (*create_buffer)(FILE *, int),
      void (*delete_buffer)(YY_BUFFER_STATE),
      YY_BUFFER_STATE (*scan_string)(const char *),
      void (*switch_to_buffer)(YY_BUFFER_STATE)):
    trunk(0),
    last_line(1),
    last_column(1),
    create_buffer(create_buffer),
    scan_string(scan_string),
    switch_to_buffer(switch_to_buffer),
    delete_buffer(delete_buffer)
{
}

int ParseTree::parseFile(const char *filename) {
  FILE *f = fopen(filename, "r");
  if (!f) {
    std::cerr << "ParseTree::parseFile(): " << filename << ": file pointer is 0\n";
    return false;
  }
  int result = parseOpenFile(filename, f);
  fclose(f);
  return result;
}

int ParseTree::parseOpenFile(const char *filename, FILE *f) {
  if (!f) {
    std::cerr << "ParseTree::parseFile(): " << filename << ": file pointer is 0\n";
    return false;
  }
  localIncludeDirs.push_back(DirName(filename));
  this->filename = filename;
  YY_BUFFER_STATE buf = create_buffer(f, YY_BUF_SIZE);
  switch_to_buffer(buf);
  return callParser(buf);
  // Don't put delete_buffer() in here, it goes in callParser and popBuffer
}

int ParseTree::parseString(const char *s) {
  this->filename = std::string("PASSED_STRING[\"") + quote(s) + "\"]";
  YY_BUFFER_STATE buf = scan_string(s);
  return callParser(buf);
  // Don't put delete_buffer() in here, it goes in callParser and popBuffer
}

void ParseTree::deleteOutputFiles() {
  for (std::vector<std::string>::iterator i = outputFiles.begin(); i != outputFiles.end(); i++) {
    //std::cout << "Removing " << *i << std::endl;
    unlink(i->c_str());
  }
}

void ParseTree::addOutputFile(const std::string &file) {
  //std::cout << "Writing " << file << std::endl;
  outputFiles.push_back(file);
}

void ParseTree::updatePosition(yyltype *loc, char *text) {
  loc->first_line = last_line;
  loc->first_column = last_column;
  for (; *text; text++) {
    if (*text == '\n') {
      last_line++;
      last_column = 1;
    }
    else if (*text == '\t')
      last_column = ((last_column - 1) | 7) + 2;
    else
      last_column++;
  }
}

void ParseTree::formatError(std::string error, const yyltype *loc,
    const std::string &val, int state) {
  formatErrorToStream(error, loc, val, state, std::cerr);
}

void ParseTree::formatErrorToStream(std::string error, const yyltype *loc,
    const std::string &val, int state, std::ostream &os) {
  os << filename << ' ' << loc->first_line << '.' << loc->first_column
    << ": \"" << val << "\": " << error << " (state " << state << ")\n" << std::flush;
}

BranchNode::BranchNode(short parserSymbol) {
  this->parserSymbol = parserSymbol;
  hidden = 0;
}

void BranchNode::pushBack(TreeNode *node) {
  v.push_back(node);
  node->parent = this;
}

void BranchNode::writeStr(std::ostream &os, int arghidden) const {
  if (!hidden)
    for (auto i: v)
      i->writeStr(os, arghidden);
}

symbolToNameFnType symbolToNameFnPtr = 0;

const char *symbolToName(int symbol) {
  if (symbolToNameFnPtr)
    return symbolToNameFnPtr(symbol);
  return "(no symbolToNameFnPtr)";
}

static int indentation = 0;

void BranchNode::writeRule(std::ostream &os) const {
  os << symbolToName(parserSymbol) << ':';
  for (unsigned i = 0; i < v.size(); i++)
    os << ' ' << symbolToName(v[i]->parserSymbol);
}

void LeafNode::writeRule(std::ostream &os) const {
  os << symbolToName(parserSymbol);
}

std::string hiddenToStr(int hidden) {
  std::string result;
  if (hidden & HIDDEN_MACRO_CALL) result.append(" (macro_call)");
  if (hidden & HIDDEN_MACRO_SUBSTITUTE) result.append(" (macro_substitute)");
  if (hidden & HIDDEN_INCLUDE_DIRECTIVE) result.append(" (include_directive)");
  if (hidden & HIDDEN_INCLUDED_CODE) result.append(" (included_code)");
  if (hidden & HIDDEN_QUOTE) result.append(" (quote)");
  return result;
}

void BranchNode::writeTree(std::ostream &os) const {
  os << hiddenToStr(hidden);
  writeRule(os);
  os << '\n';
  indentation += 2;
  for (unsigned i = 0; i < v.size(); i++) {
    os << std::string(indentation, ' ') << '$' << i + 1 << ' ';
    v[i]->writeTree(os);
  }
  indentation -= 2;
}

void LeafNode::writeTree(std::ostream &os) const {
  os << symbolToName(parserSymbol)
     << " \"" << quote(s) << '"' << hiddenToStr(hidden) << std::endl;
}

static std::set<std::string> filenamecache;
static const char *last = 0;

LeafNode::LeafNode(short parserSymbol, const char *s, const char *filename, yyltype pos): s(s) {
  //logMsg << "Constructing a LeafNode, s = '" << quote(s) << "'" << std::endl;
  this->parserSymbol = parserSymbol;
  this->pos.line = pos.first_line;
  this->pos.col = pos.first_column;
  //this->pos.filename = filename; // Unreliable
  //this->pos.filename = strdup(filename); // Waste of memory
  if (!last || strcmp(filename, last))
    last = filenamecache.insert(filename).first->c_str(); // Got it
  this->pos.filename = last;
  hidden = 0;
}

std::string TreeNode::str(int hidden) const {
  std::ostringstream oss;
  writeStr(oss, hidden);
  return oss.str();
}

std::string TreeNode::getRule() const {
  std::ostringstream oss;
  writeRule(oss);
  return oss.str();
}

void LeafNode::writeStr(std::ostream &os, int arghidden) const {
  if (arghidden & HIDDEN_QUOTE && s[0] != ' ' && s[0] != '"' && s[0] != '\''
      && s.find(' ') != std::string::npos) {
    os << '"';
    if (!(hidden & ~arghidden))
      os << s;
    os << '"';
  }
  else if (!(hidden & ~arghidden))
    os << s;
}

TreeNode *makeNode(int r1, int len, const char *const *name,
    const signed char *stos, signed char *ssp, TreeNode **vsp) {
  BranchNode *newNode = new BranchNode(r1);
  for (int i = 1; i <= len; i++)
    if (*name[stos[ssp[i - len]]] != '$'
     && *name[stos[ssp[i - len]]] != '@')
      newNode->pushBack(vsp[i - len]);
  return newNode;
}

TreeNode *makeNode(int r1, int len, const char *const *name,
    const signed char *stos, unsigned char *ssp, TreeNode **vsp) {
  BranchNode *newNode = new BranchNode(r1);
  for (int i = 1; i <= len; i++)
    if (*name[stos[ssp[i - len]]] != '$'
     && *name[stos[ssp[i - len]]] != '@')
      newNode->pushBack(vsp[i - len]);
  return newNode;
}

TreeNode *makeNode(int r1, int len, const char *const *name,
    const unsigned char *stos, short *ssp, TreeNode **vsp) {
  BranchNode *newNode = new BranchNode(r1);
  for (int i = 1; i <= len; i++)
    if (*name[stos[ssp[i - len]]] != '$'
     && *name[stos[ssp[i - len]]] != '@')
      newNode->pushBack(vsp[i - len]);
  return newNode;
}

TreeNode *makeNode(int r1, int len, const char *const *name,
    const signed char *stos, short *ssp, TreeNode **vsp) {
  BranchNode *newNode = new BranchNode(r1);
  for (int i = 1; i <= len; i++)
    if (*name[stos[ssp[i - len]]] != '$'
     && *name[stos[ssp[i - len]]] != '@')
      newNode->pushBack(vsp[i - len]);
  return newNode;
}

const TreeNode *LeafNode::subNode(unsigned i) const {
  logMsg << "Error getting subNode " << i << ". This is a LeafNode" << std::endl;
  lineabort();
}

const TreeNode *BranchNode::subNode(unsigned i) const {
  if (i < v.size())
    return v[i];
  logMsg << "Error getting subNode " << i << ". BranchNode has " << v.size() << " subNodes";
  lineabort();
}

TreeNode::TreeNode(): parent(0) {
}

const TreeNode *LeafNode::getTip() const {
  lineabort();
}

const TreeNode *BranchNode::getTip() const {
  if (v.size())
    return v[0]->getTip();
  return this;
}

const Position *BranchNode::getPosition() const {
  for (unsigned i = 0; i < v.size(); i++) {
    const Position *pos = v[i]->getPosition();
    if (pos)
      return pos;
  }
  return 0;
}

const Position *LeafNode::getPosition() const {
  return &pos;
}

std::ostream &operator << (std::ostream &os, const Position &pos) {
  return os << pos.filename << ':' << pos.line << '.' << pos.col;
}

Position::Position(): filename(""), line(0), col(0) {
}

Position::Position(const char *filename, int line, int col):
  filename(filename), line(line), col(col) {
}

