
// dep.h
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef DEP_H
#define DEP_H

#include <vector>
#include <string>
#include <map>
#include <set>
#include "AbstractProcess.h"
#include "CmdLine.h"
#include "ParseTree.h"

struct Timer;

// Are these flags properties of the file, the writing rule, or the reading rule?

                                         // file writingRule readingRule

#define FILEFLAG_PSEUDOFILE            1 // file
#define FILEFLAG_COMPARE               2 // file (or file+writingRule, but there is only one such)
#define FILEFLAG_LOCAL_MODTIME_VALID   4 // file
#define FILEFLAG_INCLUDE_MODTIME_VALID 8 // file
#define FILEFLAG_UNCHANGED            16 // file
#define FILEFLAG_FOUND_INCLUDES       32 // file
#define FILEFLAG_STAT_CHANGED         64 // file
#define FILEFLAG_CMD_CHANGED         128 // file
#define FILEFLAG_EXISTENCE           256 // file (+readingRule, but all readers use it the same way)
#define FILEFLAG_ZEROLENGTH          512 // file
#define FILEFLAG_TENTATIVE          1024 // file
#define FILEFLAG_BUILDSTATE_VALID   2048 // file
#define FILEFLAG_ALIAS              4096 // file
#define FILEFLAG_REBUILT_SAME       8192 // file
#define FILEFLAG_BUILDSTATEALL_VALID 16384 // file

#define RULEFLAG_MARKED                   1 // rule
#define RULEFLAG_REBUILD                  2 // rule
#define RULEFLAG_BUILDSTATE_VALID         4 // rule
#define RULEFLAG_IMPLHDR_VALID            8 // rule
#define RULEFLAG_COMPLETIONSECONDS_VALID 16 // rule

// build states. The values come from the min() algorithm for
// calculating state from source state.

#define BUILDSTATE_UNKNOWN       0 // not yet calculated
#define BUILDSTATE_SOURCES_STALE 1 // the file's sources are out of date; build them first
#define BUILDSTATE_STALE         2 // this file is out of date, but can be rebuilt
#define BUILDSTATE_TOUCH         3 // this file is out of date, but need only be touched
#define BUILDSTATE_BUILDING      4 // currently being built
#define BUILDSTATE_STAT_DIFF     5 // stat() differences indicate this file has changed
#define BUILDSTATE_REBUILT       6 // has been rebuilt
#define BUILDSTATE_REBUILT_SAME  7 // has been rebuilt, and has not changed
#define BUILDSTATE_OK            8 // stat() calls show this file is up to date
                                   // (BuildFile computes this from FILEFLAG_UNCHANGED)
#define BUILDSTATE_NO_SOURCES    9 // file exists, and has no source files

/*

The minimum build state of all sources tells you what you should do with a rule:

N Minimum source state  Up-to-date  Current state  Description
- --------------------  ----------  -------------  -------------------------------------
0 unknown               Don't care  unknown        you can't make a decision yet
1 stale                 Don't care  unknown        you should rebuild, but not yet
2 rebuilt-different     Don't care  stale          you should rebuild now
3 ok                    No          stale          you should rebuild now
3 ok                    Yes         ok             you should not rebuild
4 rebuilt-same          Don't care  ok             you should not rebuild
5 no sources            Don't care  ok             you should not rebuild

*/

#define VERBOSITY_NORMAL          0 // OK
#define VERBOSITY_STAT            1 // What files are being stat()ed // OK
#define VERBOSITY_READ            2 // What files are being read // OK
#define VERBOSITY_UPTODATE        3 // What rules are being skipped because up-to-date
#define VERBOSITY_UNSELECTED      4 // What rules are being skipped because not in the request tree
#define VERBOSITY_MARKED          5 // What rules are being marked for consideration
//#define VERBOSITY_RUN_DECISION   5 // Why rules are being run (the critical stat or time comparison)
#define VERBOSITY_STATE_CHANGE    6 // All changes in build state
#define VERBOSITY_START           7 // What rules are being run (usually visible from the cmd)
#define VERBOSITY_WAIT_FOR_RESULT 8
#define VERBOSITY_TRACE_RUN       9
#define VERBOSITY_FIND_GENERATED 10
#define VERBOSITY_TRACE_STATE    11
#define VERBOSITY_SHOW_ORDER     12
#define VERBOSITY_SHOW_RULES     13
#define VERBOSITY_RULE_CHANGE    14
#define VERBOSITY_PIPELINES      15
#define VERBOSITY_TIMING         16

#define VERBOSITY_ARRAY_SIZE     17

//#define FOLDERLIST_ALL      1
#define FOLDERLIST_SPECIFIC 3
#define FOLDERLIST_NONE     4

//#define RULEOUTPUT_CMDONLY 1
#define RULEOUTPUT_RESCAN  2

struct EnvVar {
  std::string name, value;
};

struct Rule;
struct RuleSet;

struct IncludesData {
  std::string filename;
  std::string workingDir;
};

struct BuildFile;

struct IncludeVector {
  std::vector<std::string> folders;
  std::vector<BuildFile *> includeVector;
  std::vector<BuildFile *> includedByVector; // inverse structure, used for invalidation
  // Note: If we are super-pedantic, the:
  // * includeModTime (currently uncached)
  // * FILEFLAG_INCLUDE_MODTIME_VALID (currently unused)
  // * fileState (can be disabled by never setting FILEFLAG_BUILDSTATE_VALID) (THIS IS
  //     THE IMPORTANT ONE)
  // also depend on the folder list.

  //struct timespec includeModTime; // Is this used? Nope

  short includeVectorState;
  short includeVectorflag;

  IncludeVector();
};

struct BuildFile {
  BuildFile(RuleSet *ruleset);
  ~BuildFile();

  int getFileState(const std::string &folderList, int mode);

  //timespec calculateIncludeModTime(const std::string &folderList);

  void calculateIncludeModTime(const std::string &folderList,
    timespec &includeModTime, timespec &rebuiltSameTime);

  void calculateLocalModTime();
  void outputTo(std::ostream &os);
  int treeAddInclude(const std::string &source, const std::string &folderList);
  void invalidateState();
  void markFile(const std::string &folderList, int cmdOrder);
  double getCompletionSeconds() const;
  int isExistenceDep() const;
  void fastAddIncludes(const TreeNode *folderList);
  void fastAddOptions(const TreeNode *optionList);

  std::string filename;
  std::vector<Rule *> readRuleVector;
  std::map<std::string, IncludeVector *> includeVectorMap;
  struct timespec localModTime, failTime, rebuiltSameTime;
  Rule *writeRule;
  RuleSet *ruleset;
  IncludesData *includesData; // for reconstruction of includes of rebuilt files
  long fileSize;
  int modTimeResult;
  float buildSeconds;
  short fileStateNone;
  short fileflag;
  short gfsRecurseGuard: 8;
  short cmtRecurseGuard: 8;
  short mfRecurseGuard: 8;
};

struct RuleRunState {
  Pipeline pipeline;
  std::map<std::string, std::string> backup; // This uses less memory than a string
                                             // in BuildFile, and is measurably faster
  unsigned runCount;
  struct timespec startTime;
};

struct Rule {
  Rule(RuleSet *ruleset);
  ~Rule();
  void addTarget(const std::string &target, const int flags, const char *sourceFile, int sourceLine);
  void addTarget(const char *target, const int flags, const char *sourceFile, int sourceLine);
  // (sourceFile and sourceLine are used to suppress some false error messages)
  int addSource(const std::string &source, const int flags);
  void addCommand(std::string command);
  int getRuleState(int debug);
  int startRule(int &resortFlag);
  void setFolder(const std::string &s);
  void markRule(int n);
  int nextCommand(int &resortFlag);
  int markRebuilt(int buildState);
  void setState(int buildState);
  void invalidateState();
  std::string ruleName() const;
  void setRunLock(const char *lock);
  timespec getFailTime() const;
  int implementsHeader();
  double getCompletionSeconds();
  void processIncludes(std::vector<std::string> *folders);
  void outputRuleTo(std::ostream &os, int flags);

  std::vector<BuildFile *> targetVector;
  std::vector<BuildFile *> sourceVector;
  std::vector<std::string> commandVector;
  std::string ruleFolder;
  std::string folderList;
  const char *runlock;
  RuleSet *ruleset;
  RuleRunState *ruleRunState;
  const Position *pos;
  float completionSeconds;
  short priority;
  short ruleflag;
  short buildstate;
  short cmdOrder;
  short savedImplementsHeader: 8;
  short ignoreExitCode: 8;
  short findOrder: 8;
};

void combinedAddIncludes(BuildFile *bf, Rule *rule, const std::string &filename,
    const std::vector<std::string> *folders, std::fstream &is, const char *workingDir,
    RuleSet *cacheset);

std::string normalizePath(std::string a);

struct IncludeRule {
  StartPattern syntax;
  std::vector<Pattern> filePatternVector;
  IncludeRule(const char *syntax);
};

struct PatternFile {
  PatternFile(RuleSet *ruleset, const char *pattern);

  BuildFile buildFile;
  Pattern pattern;
};

struct DepTree;

struct RuleSet {
  Rule *rulesetAddRule(const Position *pos);
  int processCmd(int argc, const char *argv[], File *outputFile, /*DepTree *depTree,*/ Timer &timer);
  void makeRulesFromPatterns(int argc, const char *argv[]);
  void makeRuleFromPatterns(const std::string &substitute);
  void makeRuleFromPattern(Rule *patternRule, const std::string &substitute);
  int run(File *outputFile, Timer *timer = 0);
  RuleSet();
  int waitForRuleExit(int &resortFlag, int startMore);
  void waitForAllPipelines();
  void outputTo(std::ostream &os, int outputComments);
  void checkGraph(std::ostream &os);
  BuildFile *getBuildFile(const char *filename, int flags);
  void compareToCache(const RuleSet &cache);
  void combineWithCache(const RuleSet &cache);
  std::string findFile(const std::string &filename, const std::vector<std::string> *folders,
    std::fstream &is, const char *readingFilename, const char *workingDir) const;
  void addIncludeSyntax(const std::string &s);
  void sortRuleVector(std::vector<Rule *> &ruleVector, int findOrder);
  void flattenOldFailTimes();
  void interpretVerbosity(String argVerbosity);
  void setVerbosity(int val);
  int getVerbosity(int argVerbosity, String filename) const;
  int getVerbosity(int argVerbosity, const Rule *rule) const;
  int getVerbosity(int argVerbosity, const BuildFile *buildFile) const;
  ~RuleSet();
  void processArgs(AutoCmdLine &cmdline);

  std::vector<Rule *> ruleVector;
  std::map<std::string, BuildFile *> fileMap;
  std::map<std::string, PatternFile *> patternMap;
  std::set<std::string> lockSet;
  std::map<std::string, Cmdspec *> syntaxMap;
  std::map<std::string, std::string> variableMap;
  std::vector<IncludeRule> includeRuleVector;
  File *outputFile;
  int confirmRuleChanges;
  std::ostringstream failedCommands;
  std::map<std::string, Rule *> lockMap;
  RuleSet *cacheset;
  struct timespec modTime;
  std::vector<std::string> existsExt;
  AutoCmdLine *cmdline;
  std::set<std::string> tracefileSet;
  std::string targetName;
  std::map<std::string, timespec> depFileMap;
  short optionTreeIncludes;
  short optionRuleIncludes;
  short optionDryrun;
  short optionRebuildMarked;
  short optionRebuildAll;
  short optionLink;
  short optionMkdir;
  short optionFixedOrder;
  short senseNewer;
  short senseModTime;
  short senseSize;
  short senseRule;
  short failed;
  short ignoreFail;
  short startedSomething;
  short verbosityArray[VERBOSITY_ARRAY_SIZE];
  short checkBackReferences;
  short maxPipelines;
  short maxNonIncludes;
  short readCache;
  short writeCache;
};

//void findLoop(std::vector<int> &v, std::vector<int> &result, int (*fn)(int a, int b));
std::string joinFolderAndFile(const char *folder, const std::string &file);
std::string joinFolderAndFile(const std::string &folder, const std::string &file);
std::ostream & operator << (std::ostream &os, const struct timespec &ts);

Cmdspec *getDepCmdspec();

#endif

