
// CmdLine.h
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef CMDLINE_H
#define CMDLINE_H

#include <string>
#include <vector>
#include <fstream>
#include <map>
#include "matches.h"
#include "stringify.h"
#include "ParseTree.h"

struct Option {
  std::string optionName;
  StartPattern pattern0;
  StartPattern pattern1;
  Option *primary;
  int argflag;
  Option(String optionName, int flags);
  Option();
  std::string argName;
};

#define FRAGMENTTYPE_COMMAND 1
#define FRAGMENTTYPE_OPTION  2
#define FRAGMENTTYPE_PARAM   3

#define DIVIDER_WHITESPACE 257

struct Fragment {
 const Option *option;
 const char *optionValue;
 const char *argValue;
 int fragmentType;
 char prefix;
};

std::ostream &operator <<(std::ostream &os, const Fragment &fragment);

struct Cmdspec {
  std::vector<Option> optionVector;
  std::vector<Option> paramVector;
  void outputTo(std::ostream &os, int outputComments);
  void setup(const char *spec, int flags, const char *filename, int line);
  Cmdspec(const char *spec, int flags, const Position &pos);
  int cmdspecflags;
  const char *spec;
  std::string commandName;
};

//void makeCmdspec(Cmdspec *cmdspec, const char *spec);

// There are three longevity/ownership/persistence states:
// COPY The calling function will delete the object at some time. If the called function
//      needs to retain the value, it must make a copy of the object
// FREE The calling function dynamically allocated the object specially for this call.
//      The called function must free the memory at an appropriate time
// NONE The object does not need to be copied or freed (e.g. it is a string literal)

#define KEEP_NONE 0
#define KEEP_COPY 1
#define KEEP_FREE 2

#define CMDSPEC_KEEP_GOING      1
#define CMDSPEC_FIXED          16
#define CMDSPEC_ANY_PREFIX     32
#define CMDSPEC_COUNT_SECTIONS 64

#define CMDPACK_PACK         2
#define CMDPACK_UNPACK       4
#define CMDPACK_PASSTHROUGH  8
#define CMDPACK_COPY        16

struct CmdLine {
  CmdLine(const Cmdspec *cmdspec);
  void processCmdline(int initialState, const char *cmdline, int divider);
  void processArgv(int initialState, int argc, const char *const argv[]);
  virtual void processArg(const char *arg);
  virtual void processFragment(const Fragment *fragment) = 0;
  virtual void reset(int initialState);
  void printError(std::ostream &os);
  int haltline;
  int paramIdx;
  int sectionBreak;
  int debug;
  int error;

  const char *const *argv; // Newly promoted

protected:
  const Cmdspec *cmdspec;
  const char *position;
  int paramId;
private:
  const Option *currentOption;
  std::string currentOrigStr;
  char currentPrefix;
  int state;
};

struct CmdLineRepacker: CmdLine {
  CmdLineRepacker(int flags, Cmdspec *cmdspec);
  void processArg(const char *arg);
  void processFragment(const Fragment *fragment);
  virtual char *getBack() = 0;
  virtual void setBack(char *) = 0;
  void reset(int initialState);
  virtual void outputArg(const char *arg, int keepPtr) = 0;
  int flags;
private:
  int packable;
};

struct CmdLineArgvMaker: CmdLineRepacker {
  CmdLineArgvMaker(int flags, Cmdspec *cmdspec);
  ~CmdLineArgvMaker();
  void outputArg(const char *arg, int keepPtr);
  std::vector<char *> v;
  std::vector<char *> v2;
  char *getBack();
  void setBack(char *);
};

struct CmdLineStringMaker: CmdLineRepacker {
  CmdLineStringMaker(int flags, Cmdspec *cmdspec);
  void outputArg(const char *arg, int keepPtr);
  const char *arg;
  char *getBack();
  void setBack(char *);
  const std::string &getCmdline();
private:
  std::string cmdline;
};

struct AutoCmdLine: CmdLine {
  AutoCmdLine(Cmdspec *cmdspec);
  void processFragment(const Fragment *f);

  std::map<std::string, std::vector<std::string> > optionMap;
  int sections;
  std::vector<std::pair<std::string, std::string> > optionVector;
};

void cmdlineToArgv(const char *cmdline, std::vector<char *> &argv, int divider);
std::string unpackToCmdline(Cmdspec *cmdspec,
  int argc, const char *const argv[], int stopAtFirstNonOption);
std::string packToCmdline(Cmdspec *cmdspec,
  int argc, const char *const argv[], int stopAtFirstNonOption);
std::string argvToCmdline(int argc, const char *const argv[]);

const char *fragmentTypeToStr(int fragmentType);

#endif

