
// DepTree.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "DepTree.h"
#include "depparse.h"
#include "logMsg.h"
// Extra headers (user code)
#include "quote.h"
#include <string.h>
#include "AbstractProcess.h"
#include "MacroNames.h"
#include "stringify.h"
#include <unistd.h>
#include "matches.h"
#include "clockwork.h"
#include <sys/stat.h>

DepTree *depTree; // (wizard code)

DepTree::DepTree():
    ParseTree(dep_create_buffer, dep_delete_buffer, dep_scan_string,
      dep_switch_to_buffer) { // (customizable code - constructor)
    // constructor (user code)
  activeCode = 1;
  rule = 0;
  log_lexemes = 0;
  inAMacro = 0;

#if defined __CYGWIN__ || defined _WIN32
  libExt.push_back(".dll");
#else
  libExt.push_back(".so");
#endif
  libExt.push_back(".a");
}

int DepTree::callParser(YY_BUFFER_STATE ybs) { // (wizard code)
  deplloc.first_line = 1;
  deplloc.first_column = 1;
  last_line = 1;
  last_column = 1;
  depTree = this;
  lexemes = 0;
  error = 0;
  depdebug = getenvint("DEBUG_PARSER");
  symbolToNameFnPtr = depparseGetSymbolName;
  // callParser (user code)
  int result = depparse(); // (wizard code)
  dep_delete_buffer(ybs);
  if (!result)
    result = error;
  // Cleanup code (user code)
  return result; // (wizard code)
}

int DepTree::processToken(int token) {
  ParseTree::updatePosition(&deplloc, deptext);
  static int log_lexemes = getenvint("LOG_LEXEMES");
  if (log_lexemes || this->log_lexemes) {
    std::string symbolName = depparseGetSymbolName(depparseGetSymbolNumber(token));
    if (ruleset->outputFile->getOStream())
      // Do both. One for testing, one for debugging
      *ruleset->outputFile->getOStream() << symbolName << " (" << filename << ' '
        << deplloc.first_line << '.' << deplloc.first_column << ":"
        << " \"" << quote(deptext) << "\")\n";
    if (log_lexemes)
      logMsg << symbolName << " (" << filename << ' '
        << deplloc.first_line << '.' << deplloc.first_column << ":"
        << " \"" << quote(deptext) << "\")\n";
  }
  lexemes++;
  deplval = new LeafNode(depparseGetSymbolNumber(token), deptext, filename.c_str(), deplloc); // (customizable code - new LeafNode)
  return token;
}

int DepTree::popBuffer() {
  // (user code)
  if (!sourceFileVector.size()) {
    return 1;
  }
  if (deplexGetCurrentFile()) {
    fclose(deplexGetCurrentFile());
  }
  dep_delete_buffer(deplexGetCurrentBuffer());
  dep_switch_to_buffer(sourceFileVector.back().bufferState);
  filename = sourceFileVector.back().filename;
  if (ruleset->getVerbosity(VERBOSITY_READ, filename))
    if (!inAMacro)
      *ruleset->outputFile->getOStream() << "Returning to " << filename << std::endl;
  depReaderCurrentFolder = sourceFileVector.back().folder;
  deplloc = sourceFileVector.back().location;
  last_line = sourceFileVector.back().location.first_line;
  last_column = sourceFileVector.back().location.first_column;
  inAMacro = sourceFileVector.back().inAMacro;
  sourceFileVector.pop_back();
  return 0;
}

void DepTree::callMacro(const TreeNode *node) {
  std::string str(node->str());
  //static int defect = getenvint("DEFECT", 0);
  if (str == "LINE" /*&& defect != 13*/) {
    pushBuffer(filename + " (macro " + str + ")", 1);
    dep_scan_string(stringify(last_line).c_str());
  }
  else if (variableMap.count(str)) {
    pushBuffer(filename + " (macro " + str + ")", 1);
    dep_scan_string(variableMap[str].c_str());
  }
  else {
    const char *value = getenv(str.c_str());
    if (value) {
      pushBuffer(filename + " env " + str, 1);
      //logMsg << "Calling an env" << std::endl;
      dep_scan_string(value);
    }
  }
}

void DepTree::substituteCommand(const TreeNode *node) {
  std::string command(node->str());
  Process process(command);
  process.pipeOutput();
  process.run();

  pushBuffer(filename + " cmdsubst " + command, 1);
  std::string output;
  ReadWholeFile(process.outputPipe, output);
  dep_scan_string(output.c_str());
}

void DepTree::includeFile(const TreeNode *node, int nocd) {
  if (!activeCode)
    return;
  // Keep good track of the filenames!
  //logMsg << logValue(depReaderCurrentFolder) << std::endl;
  //logMsg << logValue(node->str()) << std::endl;
  std::string filename = joinFolderAndFile(depReaderCurrentFolder, node->str());
  if (ruleset->depFileMap.count(filename)) {
    //logMsg << "Have already read " << filename << std::endl;
    static int includeOnce = getenvint("INCLUDE_ONCE", 1);
    if (includeOnce)
      return;
  }
  if (ruleset->getVerbosity(VERBOSITY_READ, filename))
    *ruleset->outputFile->getOStream() << "Including " << filename << std::endl;
  FILE *file = fopen(filename.c_str(), "r");
  if (file) {
    struct stat statbuf;
    fstat(fileno(file), &statbuf);
    ruleset->depFileMap[filename] = statbuf.st_mtim;
    //logMsg << logValue(filename) << ' ' << logValue(statbuf.st_mtim) << std::endl;

    pushBuffer(this->filename + ':' + stringify(last_line) + " include of " + filename, 0);
    if (!nocd) {
      depReaderCurrentFolder = DirName(filename.c_str());
    }
    dep_switch_to_buffer(dep_create_buffer(file, YY_BUF_SIZE));
  }
  else {
    int err = errno;
    logMsg << "dep: cannot read " << filename;
    if (err == ENOENT)
      std::cerr << " File does not exist" << std::endl;
    else {
      std::cerr << " error is " << ErrnoToMacroName(err) << ' ' << strerror(err) << std::endl;
    }
  }
}

// Some day: Move as much of this as possible to ParseTree

void DepTree::pushBuffer(const std::string &newFilename, int inAMacro) {
  SourceFile sourceFile;
  sourceFile.bufferState = deplexGetCurrentBuffer();
  sourceFile.filename = filename;
  sourceFile.location.first_line = last_line;
  sourceFile.location.first_column = last_column;
  sourceFile.inAMacro = this->inAMacro;
  sourceFile.folder = depReaderCurrentFolder;
  sourceFileVector.push_back(sourceFile);
  filename = newFilename;
  this->inAMacro = inAMacro;
  //logMsg << logValue(deplloc.first_line) << std::endl;
  if (!inAMacro) {
    deplloc.first_line = 1;
    deplloc.first_column = 1;
    last_line = 1;
    last_column = 1;
  }
  //logMsg << logValue(newFilename) << std::endl;
}

void DepTree::setVariable(const TreeNode *variable, const TreeNode *value, int env) {
  if (activeCode) {
    variableMap[variable->str()] = value->str();
    if (variable->str() == "line_") {
      last_line = atoi(value->str().c_str());
      //logMsg << "Changing line to " << last_line << std::endl;
    }
    if (env)
      setenv(variable->str().c_str(), value->str().c_str(), 1);
  }
}

int DepTree::evaluateExpr(const TreeNode *node) {
  std::string grammarRule = node->getRule();
  int result;
  if (grammarRule == "expr: expr BARBAR ws.opt expr")
    result = evaluateExpr(node->subNode(0)) || evaluateExpr(node->subNode(3));
  else if (grammarRule == "expr: expr '=' ws.opt expr") {
    result = stripWhitespace(node->subNode(0)->str()) == stripWhitespace(node->subNode(3)->str());
  }
  else if (grammarRule == "expr: '=' ws.opt expr")
    result = node->subNode(2)->str() == "";
  else if (grammarRule == "expr: '!' ws.opt expr")
    result = !evaluateExpr(node->subNode(2));
  else if (grammarRule == "expr: '!' ws.opt")
    result = 1;
  else if (grammarRule == "expr: argument ws.opt")
    result = evaluateExpr(node->subNode(0));
  else if (grammarRule == "argument: argumentPiece")
    result = evaluateExpr(node->subNode(0));
  else if (grammarRule == "argumentPiece: EVERYTHINGELSE")
    result = evaluateExpr(node->subNode(0));
  else if (grammarRule == "EVERYTHINGELSE") {
    result = atoi(node->str().c_str());
  }
  else {
    logMsg << "Don't understand " << logValue(grammarRule) << " at " << *node->getPosition()
      << std::endl;
    lineabort();
  }
  return result;
}

void DepTree::expandForLoop(const Position *pos, const std::string &identifier,
    const TreeNode *node, const TreeNode *ws, const TreeNode *cmd, std::ostringstream &os) {
  std::string grammarRule = node->getRule();
  if (grammarRule == "argumentList: argumentList argumentPiece ws.opt") {
    expandForLoop(pos, identifier, node->subNode(0), ws, cmd, os);
    expandForLoop(pos, identifier, node->subNode(1), ws, cmd, os);
  }
  else if (grammarRule == "argumentList:");
  else if (grammarRule == "argumentPiece: IDENTIFIER"
        || grammarRule == "argumentPiece: STRING") {
    std::string backup;
    int count = variableMap.count(identifier);
    if (count)
      backup = variableMap[identifier];

    // This is a whitespace-separated list expander
    std::string value1 = node->str(HIDDEN_MACRO_CALL);
    std::string value;
    skipAndDequote(value1.c_str(), &value);
    std::string dequoted;
    const char *s = value.c_str();
    while (s && *s) {
      s = skipAndDequote(s, &dequoted);
      os << identifier << '=' << dequoted << '\n';
      os << "line_=" << pos->line << '\n';
      os << ws->str(HIDDEN_MACRO_CALL);
      os << cmd->str(HIDDEN_MACRO_CALL);
      s = skipWhitespace(s);
    }

    if (count)
      os << identifier << '=' << backup << '\n';
    else
      os << identifier << "=\n";
  }
  else {
    logMsg << "Don't understand " << logValue(grammarRule) << std::endl;
    lineabort();
  }
}

const char *fragmentTypeToMacroName(int t) {
  if (t == FRAGMENTTYPE_OPTION) return "OPTION";
  if (t == FRAGMENTTYPE_PARAM) return "PARAM";
  if (t == FRAGMENTTYPE_COMMAND) return "PARAM";
  return "(Unknown fragment type)";
}

struct DepCmdLine: CmdLine {
  DepCmdLine(const char *currentFolder, Rule *rule):
    CmdLine(0), flag(0), paramType(0), depline(0),
      rule(rule), currentFolder(currentFolder), dclFilename("b1ab1a") {
  }
  int flag;
  int paramType;
  void processFragment(const Fragment *fragment) {
    //logMsg << logValue(dclFilename) << std::endl;
    //logMsg << fragment->optionValue << ' ' << fragmentTypeToMacroName(fragment->fragmentType) << ' '
      //<< !!fragment->option << ' ' << (fragment->argValue? fragment->argValue: "(null)")
      //<< ' ' << logValue(depline) << std::endl;
    if (fragment->fragmentType == FRAGMENTTYPE_OPTION
     || fragment->fragmentType == FRAGMENTTYPE_PARAM && depline == 0) {
      if (fragment->fragmentType == FRAGMENTTYPE_PARAM) {
        //logMsg << logValue(fragment->argValue) << ' ' << logValue(paramType) << std::endl;
        if (!strcmp(fragment->argValue, "%in"))
          paramType = 1;
        else if (!strcmp(fragment->argValue, "%out"))
          paramType = 2;
        else if (!strcmp(fragment->argValue, ">"))
          paramType = 2;
        else if (!strcmp(fragment->argValue, "%cmpout"))
          paramType = 3;
        else if (!strcmp(fragment->argValue, "%pseudo"))
          paramType = 4;
        else if (!strcmp(fragment->argValue, "%exists"))
          paramType = 5;
        else if (!strcmp(fragment->argValue, "%outz"))
          paramType = 6;
        else if (!strcmp(fragment->argValue, "%lock")) {
          //logMsg << "Setting paramType to 7" << std::endl;
          paramType = 7;
        }
        else if (!strcmp(fragment->argValue, "%priority")) {
          rule->priority = 1;
          //logMsg << "Setting paramType to 8" << std::endl;
          paramType = 8;
        }
        else if (!strcmp(fragment->argValue, "%alias"))
          paramType = 9;
        else if (paramType == 1)
          rule->addSource(fragment->argValue, 0);
        else if (paramType == 2)
          rule->addTarget(fragment->argValue, 0, dclFilename, dclLine);
        else if (paramType == 3)
          rule->addTarget(fragment->argValue, FILEFLAG_COMPARE, dclFilename, dclLine);
        else if (paramType == 4)
          rule->addTarget(fragment->argValue, FILEFLAG_PSEUDOFILE, dclFilename, dclLine);
        else if (paramType == 5)
          rule->addSource(fragment->argValue, FILEFLAG_EXISTENCE);
        else if (paramType == 6)
          rule->addTarget(fragment->argValue, FILEFLAG_ZEROLENGTH, dclFilename, dclLine);
        else if (paramType == 7) {
          //logMsg << "There's that paramType 7" << std::endl;
          rule->setRunLock(fragment->argValue);
        }
        //else if (paramType == 8) {
          //logMsg << "There's that paramType 8" << std::endl;
          //rule->priority = 1;
        //}
        else if (paramType == 9)
          rule->addTarget(fragment->argValue, FILEFLAG_ALIAS, dclFilename, dclLine);
      }
      if (paramType == 0 && fragment->option) {
        //logMsg << logValue(*fragment) << std::endl;
        if (fragment->option->primary->argName == "%includedir") {
          //logMsg << "Found a %includedir" << std::endl;
          includeDirList.push_back(fragment->argValue);
        }
        else if (fragment->option->primary->argName == "%libdir") {
          // Speedup: eliminate values that are absolute paths (this works because they
          // are all system include folders such as /usr/lib64/pgsql96, and system includes
          // are ignored anyway)
          if (fragment->argValue)
            if (*fragment->argValue != '/')
              libDirList.push_back(fragment->argValue);
            else {
              //logMsg << logValue(fragment->argValue) << std::endl;
              //lineabort();
            }
        }
        else if (fragment->option->primary->argName == "%library") {
          libFileList.push_back(fragment->argValue);
        }
        else if (fragment->option->primary->argName == "%in") {
          rule->addSource(fragment->argValue, 0);
        }
        else if (fragment->option->primary->argName == "%exists") {
          rule->addSource(fragment->argValue, FILEFLAG_EXISTENCE);
        }
        else if (fragment->option->primary->argName == "%out") {
          //logMsg << "Calling addTarget" << std::endl;
          //logMsg << logValue(rule) << std::endl;
          //logMsg << logValue(rule->ruleName()) << std::endl;
          //logMsg << logValue(fragment) << std::endl;
          //logMsg << logValue(fragment->optionValue) << std::endl;
          //logMsg << logValue((void *)fragment->argValue) << std::endl;

          ////logMsg << logValue(fragment->argValue) << std::endl;

          rule->addTarget(fragment->argValue, 0, dclFilename, dclLine);
          //logMsg << "Called addTarget" << std::endl;
        }
        else if (fragment->option->primary->argName == "%cmpout") {
          rule->addTarget(fragment->argValue, FILEFLAG_COMPARE, dclFilename, dclLine);
        }
        else if (fragment->option->primary->argName == "%outz") {
          rule->addTarget(fragment->argValue, FILEFLAG_ZEROLENGTH, dclFilename, dclLine);
        }
        else if (fragment->option->primary->argName == "%pseudo") {
          rule->addTarget(fragment->argValue, FILEFLAG_PSEUDOFILE, dclFilename, dclLine);
        }
        else if (fragment->option->primary->argName == "%alias") {
          rule->addTarget(fragment->argValue, FILEFLAG_ALIAS, dclFilename, dclLine);
        }
      }
    }
    else if (fragment->fragmentType == FRAGMENTTYPE_PARAM) {
      if (depline == 0) {
        lineabort();
      }
      else if (depline) {
        if (!strcmp(fragment->argValue, "%in"))
          depline = 1;
        else if (!strcmp(fragment->argValue, "%out"))
          depline = 2;
        else if (!strcmp(fragment->argValue, "%cmpout"))
          depline = 3;
        else if (!strcmp(fragment->argValue, "%pseudo"))
          depline = 4;
        else if (!strcmp(fragment->argValue, "%exists"))
          depline = 5;
        else if (!strcmp(fragment->argValue, "%outz"))
          depline = 6;
        else if (!strcmp(fragment->argValue, "%lock"))
          depline = 7;
        else if (!strcmp(fragment->argValue, "%alias"))
          depline = 9;
        else if (!strcmp(fragment->argValue, "%priority")) {
          rule->priority = 1;
          depline = 0;
        }
        else if (depline == 1)
          rule->addSource(fragment->argValue, 0);
        else if (depline == 2)
          rule->addTarget(fragment->argValue, 0, dclFilename, dclLine);
        else if (depline == 3)
          rule->addTarget(fragment->argValue, FILEFLAG_COMPARE, dclFilename, dclLine);
        else if (depline == 4)
          rule->addTarget(fragment->argValue, FILEFLAG_PSEUDOFILE, dclFilename, dclLine);
        else if (depline == 5)
          rule->addSource(fragment->argValue, FILEFLAG_EXISTENCE);
        else if (depline == 6)
          rule->addTarget(fragment->argValue, FILEFLAG_ZEROLENGTH, dclFilename, dclLine);
        else if (depline == 7)
          rule->setRunLock(fragment->argValue);
        //else if (depline == 8)
          //rule->priority = 1;
        else if (depline == 9)
          rule->addTarget(fragment->argValue, FILEFLAG_ALIAS, dclFilename, dclLine);
        else
          lineabort();
      }
    }
    else if (fragment->fragmentType == FRAGMENTTYPE_COMMAND) {
      if (!strcmp(fragment->argValue, "%in")) {
        depline = 1;
        flag = 0;
      }
      else if (!strcmp(fragment->argValue, "%out")) {
        depline = 2;
        flag = 0;
      }
      else if (!strcmp(fragment->argValue, "%cmpout")) {
        depline = 3;
        flag = FILEFLAG_COMPARE;
      }
      else if (!strcmp(fragment->argValue, "%pseudo")) {
        depline = 4;
        flag = FILEFLAG_PSEUDOFILE;
      }
      else if (!strcmp(fragment->argValue, "%exists")) {
        depline = 5;
        flag = FILEFLAG_EXISTENCE;
      }
      else if (!strcmp(fragment->argValue, "%outz")) {
        depline = 6;
        flag = FILEFLAG_ZEROLENGTH;
      }
      else if (!strcmp(fragment->argValue, "%lock")) {
        depline = 7;
        lineabort();
      }
      else if (!strcmp(fragment->argValue, "%alias")) {
        depline = 9;
        flag = FILEFLAG_ALIAS;
      }
      else if (!strcmp(fragment->argValue, "export"));
      else {
        std::string modifiedArgValue;
        if (fragment->argValue[0] == '-')
          modifiedArgValue = &fragment->argValue[1];
        else
          modifiedArgValue = fragment->argValue;
        if (depTree->ruleset->syntaxMap.count(modifiedArgValue)) {
          cmdspec = depTree->ruleset->syntaxMap[modifiedArgValue];
        }
        else {
          if (rule->ruleset->getVerbosity(VERBOSITY_NORMAL, fragment->argValue))
            errorLine = std::string("dep: ") + dclFilename + " " + stringify(dclLine)
              + ": unknown syntax for " + fragment->argValue;
            cmdspec = 0;
        }
      }
    }
  }
  std::vector<std::string> includeDirList;
  std::vector<std::string> libDirList;
  std::vector<std::string> libFileList;
  int depline;
  Rule *rule;
  const char *currentFolder;
  std::string errorLine;
  const char *dclFilename;
  int dclLine;
};

// Read summarized file information from a .DepFileCache, including their "include" files

void DepTree::fastAddFile(const std::string &filename, const TreeNode *buildState,
    const TreeNode *optionList, const TreeNode *folderList) {

  BuildFile *buildFile = ruleset->getBuildFile(filename.c_str(), 0);

  /*std::string buildStatusStr = buildStatus->str();
  if (buildStatusStr == "REBUILT_SAME") {
    static int enablePersistentSame = getenvint("PERSISTENT_SAME", 1);
    //logMsg << logValue(enablePersistentSame) << std::endl;
    if (enablePersistentSame) {
      //buildFile->fileflag |= FILEFLAG_REBUILT_SAME;
      //logMsg << "Setting FILEFLAG_REBUILT_SAME on " << filename << '\n';
    }
  }*/
  buildFile->fastAddOptions(optionList);
  buildFile->fastAddIncludes(folderList);
}

void BuildFile::fastAddOptions(const TreeNode *node) {
  std::string grammarRule = node->getRule();
  if (grammarRule == "fileInfoOptionList:");
  else if (grammarRule == "fileInfoOptionList: fileInfoOptionList fileInfoOption newline") {
    fastAddOptions(node->subNode(0));
    fastAddOptions(node->subNode(1));
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_BUILDSECONDS ws.opt EVERYTHINGELSE ws.opt") {
    buildSeconds = atof(node->subNode(2)->str().c_str());
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_FILESIZE ws.opt EVERYTHINGELSE ws.opt") {
    fileSize = atol(node->subNode(2)->str().c_str());
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_FAILTIME ws.opt timestamp") {
    std::string timestampRule = node->subNode(2)->getRule();
    if (timestampRule == "timestamp: EVERYTHINGELSE ws.opt EVERYTHINGELSE ws.opt") {
      std::string ts = node->subNode(2)->subNode(0)->str() + " "
                     + node->subNode(2)->subNode(2)->str();
      int year, mon, day, hour, min;
      Decimal sec;
      std::string tzname, tz;
      scanDatetime("y-m-d h:m:sz", ts.c_str(), &year, &mon, &day,
        &hour, &min, &sec, &tzname, &tz);
      changeOffset(&year, &mon, &day, &hour, &min, &sec, &tzname, &tz, "UTC", "+0");
      encodeDatetime(year, mon, day, hour, min, (int)sec,
        UNIX_EPOCH, 60, &failTime.tv_sec);
      failTime.tv_nsec = (sec - (int)sec) * 1000000000;
      //logMsg << logValue(failTime) << std::endl;
    }
    else {
      logMsg << logValue(timestampRule) << std::endl;
      logMsg << logValue(node->subNode(2)->str()) << std::endl;
      lineabort();
    }
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_CHANGETIME ws.opt IDENTIFIER ws.opt") {
    std::string ts = node->subNode(2)->str();
    if (ts == "unknown") {
      fileflag &= ~FILEFLAG_LOCAL_MODTIME_VALID;
      modTimeResult = 0;
    }
    else if (ts == "ENOENT") {
      fileflag |= FILEFLAG_LOCAL_MODTIME_VALID;
      modTimeResult = ENOENT;
    }
    else if (ts == "EINVAL") {
      fileflag |= FILEFLAG_LOCAL_MODTIME_VALID;
      modTimeResult = EINVAL;
    }
    else {
      lineabort();
    }
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_CHANGETIME ws.opt timestamp") {
    std::string ts = node->subNode(2)->str();
    int year, mon, day, hour, min;
    Decimal sec;
    std::string tzname, tz;
    scanDatetime("y-m-d h:m:sz", ts.c_str(), &year, &mon, &day,
      &hour, &min, &sec, &tzname, &tz);
    changeOffset(&year, &mon, &day, &hour, &min, &sec, &tzname, &tz, "UTC", "+0");
    encodeDatetime(year, mon, day, hour, min, (int)sec,
      UNIX_EPOCH, 60, &localModTime.tv_sec);
    localModTime.tv_nsec = (sec - (int)sec) * 1000000000;
    fileflag |= FILEFLAG_LOCAL_MODTIME_VALID;
    modTimeResult = 0;
    //logMsg << "Setting modTimeResult to 0 and setting MODTIME_VALID" << std::endl;
  }
  else if (grammarRule == "fileInfoOption: KEYWORD_REBUILTSAMETIME ws.opt timestamp") {
    std::string ts = node->subNode(2)->str();
    int year, mon, day, hour, min;
    Decimal sec;
    std::string tzname, tz;
    scanDatetime("y-m-d h:m:sz", ts.c_str(), &year, &mon, &day,
      &hour, &min, &sec, &tzname, &tz);
    changeOffset(&year, &mon, &day, &hour, &min, &sec, &tzname, &tz, "UTC", "+0");
    encodeDatetime(year, mon, day, hour, min, (int)sec,
      UNIX_EPOCH, 60, &rebuiltSameTime.tv_sec);
    rebuiltSameTime.tv_nsec = (sec - (int)sec) * 1000000000;
    fileflag |= FILEFLAG_LOCAL_MODTIME_VALID;
    modTimeResult = 0;
    //logMsg << filename << ' ' << logValue(localModTime) << std::endl;
    //logMsg << filename << ' ' << logValue(rebuiltSameTime) << std::endl;
    //lineabort();
  }
  else {
    logMsg << "Don't understand " << logValue(grammarRule) << std::endl;
    lineabort();
  }
}

// fastAddOptions() - process extra buildfile lines
// fastAddIncludes() - process include directives parsed from a .DepFileCache file
// Change these to methods of BuildFile DONE

void BuildFile::fastAddIncludes(const TreeNode *node) {
  std::string grammarRule = node->getRule();
  if (grammarRule == "includes:");
  else if (grammarRule == "includes: includes KEYWORD_FOLDERS ws.opt"
      " argumentList newline KEYWORD_INCLUDES ws.opt argumentList newline") {
    fastAddIncludes(node->subNode(0));

    std::string folderList = node->subNode(3)->str();
    std::string folderList2 = folderList;
    IncludeVector *newIncludeVector = new IncludeVector;
    includeVectorMap[folderList] = newIncludeVector;
    while (folderList.size()) {
      std::string field = GetField(folderList, " ");
      newIncludeVector->folders.push_back(field);
    }

    Cmdspec cmdspec("", 0, *node->subNode(7)->getPosition());
    CmdLineArgvMaker cmdline(CMDPACK_UNPACK, &cmdspec);
    cmdline.processCmdline(FRAGMENTTYPE_PARAM, node->subNode(7)->str().c_str(), ' ');
    for (std::vector<char *>::iterator i = cmdline.v.begin(); i != cmdline.v.end(); i++) {
      //logMsg << "Adding an include" << std::endl;
      treeAddInclude(*i, folderList2);
    }
  }
  else {
    logMsg << "Don't understand " << logValue(grammarRule) << std::endl;
    lineabort();
  }
}

void DepTree::depTreeAddRule(const Position *pos) {
  //logMsg << "Adding rule " << std::endl;
  rule = ruleset->rulesetAddRule(pos);
  //lineabort();
  rule->setFolder(currentFolder);
}

void DepTree::depTreeAddCommand(std::string s/*, const Position *pos*/) {
  //logMsg << "Adding command " << logValue(s) << std::endl;
  size_t posn = s.find("#");
  if (posn != std::string::npos) {
    //logMsg << "Changing " << s << std::endl;
    s.resize(posn);
    //logMsg << "      to " << s << std::endl;
  }
  if (!rule)
    lineabort();
  for (;;) {
    size_t pos = s.find("\\\n");
    if (pos == std::string::npos)
      break;
    // Ways to handle backslash-newlines:
    // 1) erase them at the lexer
    // 2) erase them here (this is the current way)
    // 3) handle them specially in the argument scanner
    s.erase(pos, 2);
  }
  if (!s.size())
    return;

  //logMsg << logValue(lookForDependencies) << std::endl;

  // XXX lookForDependencies is not properly calculated. TODO: Calculate better DONE
  if (lookForDependencies) {

    //*ruleset->outputFile->getOStream() << "Constructing a DepCmdLine " logValue(s) << std::endl;
    DepCmdLine cmdline(depReaderCurrentFolder.c_str(), rule);

    //cmdline.dclFilename = pos->filename;
    //cmdline.dclLine = pos->line;
    cmdline.dclFilename = rule->pos->filename;
    cmdline.dclLine = rule->pos->line;

    cmdline.processCmdline(FRAGMENTTYPE_COMMAND, s.c_str(), DIVIDER_WHITESPACE);
    if (cmdline.errorLine.size() && !cmdline.paramType) {
      if (ruleset->outputFile->getOStream())
        *ruleset->outputFile->getOStream() << cmdline.errorLine << std::endl;
    }
    rule->addCommand(s);

    if ((s[0] != '%')) {
      cmdline.includeDirList.push_back(""); // Put the empty include at the end of the list

      rule->processIncludes(&cmdline.includeDirList);
    }

    std::string fullName;
    for (auto j: cmdline.libFileList) {
      int foundIt = 0;
      for (auto i: cmdline.libDirList) {
        for (auto k: libExt) {
          fullName = joinFolderAndFile(i, "lib" + j + k);
          if (ruleset->fileMap.count(fullName)
              && !(ruleset->fileMap[fullName]->fileflag & FILEFLAG_TENTATIVE)) {
            rule->addSource(fullName, 0);
            foundIt = 1;
            break;
          }
        }
      }
      if (!foundIt) {
        for (auto i: cmdline.libDirList) {
          for (auto k: libExt) {
            fullName = joinFolderAndFile(i, "lib" + j + k);
            //if (!ruleset->fileMap.count(fullName)) {
              rule->addSource(fullName, FILEFLAG_TENTATIVE);
            //}
          }
        }
      }
    }
  }
  else {
    //logMsg << "lookForDependencies is 0 so not looking" << std::endl;
    rule->addCommand(s);
  }
  //logMsg << "Added command " << logValue(s) << std::endl;
}

int DepTree::readFile(String filename) {
  if (ruleset->maxNonIncludes == -13)
    ruleset->maxNonIncludes = 1000;

  depTree = this;
  FILE *f = fopen(filename, "r");
  if (!f) {
    int err = errno;
    if (ruleset->getVerbosity(VERBOSITY_READ, filename))
      *ruleset->outputFile->getOStream() << "Couldn't open " << filename.s << " for reading "
      << logValue(err) << ' ' << ErrnoToMacroName(err)
      << std::endl;
    return -1;
  }

  struct stat statbuf;
  fstat(fileno(f), &statbuf);
  ruleset->modTime = statbuf.st_mtim;
  //logMsg << logValue(statbuf.st_mtim) << std::endl;

  if (ruleset->getVerbosity(VERBOSITY_READ, filename))
    *ruleset->outputFile->getOStream() << "Opened " << filename.s << " for reading" << std::endl;
  int result = parseOpenFile(filename, f);
  fclose(f);
  int deplex_destroy(void);
  deplex_destroy();
  if (result) {
    logMsg << logValue(result) << std::endl;
    return -1;
  }
  return 0;
}

int DepTree::runImmediate(String s) {
  //if (!strncmp(s.s, "imm echo ", 9)) {
  if (!strncmp(s.s, "print ", 6)) {
    //std::cout << s.s + 9 << '\n'; // Skip over the words "imm echo"
    *ruleset->outputFile->getOStream() << s.s + 6 << '\n';
    return 1;
  }
  return 0;
}

void removeComment(std::string &s) {
  size_t hash = s.find('#');
  if (hash != std::string::npos) {
    //logMsg << logValue(s) << std::endl;
    s.resize(hash);
    s = stripWhitespace(s);
    //logMsg << logValue(s) << std::endl;
  }
}

void DepTree::addExistsExts(const TreeNode *node) {
  std::string grammarRule = node->getRule();
  if (grammarRule == "argumentList: argumentList argumentPiece ws.opt") {
    addExistsExts(node->subNode(0));
    addExistsExts(node->subNode(1));
  }
  else if (grammarRule == "argumentList:");
  else if (grammarRule == "argumentPiece: EVERYTHINGELSE") {
    ruleset->existsExt.push_back(node->subNode(0)->str());
  }
  else if (grammarRule == "argumentPiece: IDENTIFIER") {
    ruleset->existsExt.push_back(node->subNode(0)->str());
  }
  else {
    logMsg << logValue(grammarRule) << std::endl;
    lineabort();
  }
}

void DepTree::addOption(const std::string &s) {
  if (activeCode) {
    //logMsg << logValue(s) << std::endl;
    //ruleset->cmdline->debug = 1;
    int paramIdx = ruleset->cmdline->paramIdx;
    ruleset->cmdline->processCmdline(FRAGMENTTYPE_OPTION, s.c_str(), DIVIDER_WHITESPACE);
    ruleset->cmdline->paramIdx = paramIdx;
    //ruleset->cmdline->debug = 0;
    ruleset->processArgs(*ruleset->cmdline);
    //logMsg << "checkpoint" << std::endl;
  }
}

