
// cross_platform.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "cross_platform.h"
#include <stdio.h>
#include "logMsg.h"
#include "MacroNames.h"
#include "interproc.h"

#if !defined _WIN32 && !defined __CYGWIN__

#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>
#include <iostream>
#include "Utils.h"

LPSTR GetCommandLine() {
  static std::string result;
  static int initialised = 0;
  if (!initialised) {
    FILE *f = fopen("/proc/self/cmdline", "r");
    result = ReadWholeFile(f, result);
    for (unsigned i = 0; i < result.size(); i++)
      if (!result[i])
        result[i] = ' ';
    initialised = 1;
  }
  std::cout << "command line is " << result << std::endl;
  return (char *)result.c_str();
}

int WriteFile(HANDLE h, const void *buffer, DWORD size, DWORD * written, void *overlapped) {
  return write(h, buffer, size);
}

DWORD GetTickCount() {
  struct timeval tv;
  gettimeofday(&tv, 0);
  return tv.tv_usec / 1000 + tv.tv_sec * 1000;
}

HANDLE CreateFile(LPCSTR filename, DWORD desiredAccess, DWORD /*shareMode*/, void * /*security*/, DWORD createFlags, DWORD /*flags*/, HANDLE /*template*/) {
  std::cerr << logValue(filename) << std::endl;
  int accessMode = 0;
  if (desiredAccess & FILE_READ_DATA)
    if (desiredAccess & FILE_WRITE_DATA)
      accessMode = O_RDWR;
    else
      accessMode = O_RDONLY;
  else
    if (desiredAccess & FILE_WRITE_DATA)
      accessMode = O_WRONLY;
    else
      return 0;
  if (createFlags == CREATE_ALWAYS)
    accessMode |= O_CREAT;
  else if (createFlags == OPEN_EXISTING);
  else
    lineabort();
  int fd = open(filename, accessMode, 0666);
  return fd;
}

#endif

#if defined _WIN32 && !defined __CYGWIN__

#include <windows.h>

int unsetenv(const char *var) {
  lineabort();
}

int setenv(const char *var, const char *val, int overwrite) {
  lineabort();
}

void sleep(int a) {
  Sleep(1000 * a);
}

#endif

#if !defined _WIN32 && !defined __CYGWIN__ && !defined __MINGW32__

void Sleep(int a) {
  //logMsg << "Sleeping " << a << " millis" << std::endl;
  Timer t;
  struct timespec ts;
  ts.tv_sec = a / 1000;
  ts.tv_nsec = (a % 1000) * 1000000;
  nanosleep(&ts, 0);
  //logMsg << "Slept " << a << "millis " << logValue(t.getMilliseconds()) << std::endl;
  //lineabort();
}

#endif

void (*cleanup)() = 0;

void lineabort_f(const char *file, int line, const char *fn) {
  if (cleanup)
    cleanup();
  // abort() causes a segfault during static initialisation in Cygwin.
  // When this happens, gdb cannot produce a back trace (even when it
  // runs the program, and the *.exe.stackdump file is as always quite
  // unusable)

  fprintf(stderr, "lineabort() called from %s() at %s %d\n", fn, BaseName(file), line);
  fflush(stderr); // this is required in mingw32 for some reason

  if (globalShare) {
    //logMsg << "Setting stopnow" << std::endl;
    globalShare->stopnow = 1;
  }

#ifdef __MINGW32__
  *(int *)0 = 123;  // This ends up saying 'Segmentation fault'. But, gdb can see it
  //raise(SIGABRT); // No error message but gdb can't see it
  //abort();        // VERY annoying error message:
                    // "This application has requested the Runtime terminate it in an unusual way"
                    // "... Please contact the application's support team ...".
                    // Also, gdb says "no stack".
#endif

  //logMsg << "checkpoint" << std::endl;
  abort();
}

void lineexit_f(const char *fn, const char *file, int line, int exitcode) {
  if (cleanup)
    cleanup();
  // abort() causes a segfault during static initialisation in Cygwin.
  // When this happens, gdb cannot produce a back trace (even when it
  // runs the program, and the *.exe.stackdump file is as always quite
  // unusable)

  fprintf(stderr, "lineexit() called from %s() at %s %d\n", fn, BaseName(file), line);
  fflush(stderr); // this is required in mingw32 for some reason

  exit(exitcode);
}

// This is a redirector for Linux only. Windows and Cygwin have a perfectly good one

#if !defined __CYGWIN__ && !defined _WIN32
const char *WinErrorToMacroName(unsigned long n) {
  return ErrnoToMacroName(n);
}
#endif

