
// depmain.cpp
// Copyright 2022 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"
#include "logMsg.h"
#include <fstream>
#include "dep.h"
#include <string.h>
#include "MacroNames.h"
#include <sys/stat.h>
#include "AbstractProcess.h"
#include "matches.h"
#include "DepTree.h"
#include "argv.h"

const char *bs = "dep version 0.6 copyright 2022 Matthew Rickard. Buildstamp " __DATE__ " " __TIME__;

int main(int argc, const char *argv[]) {
  olib::argc = argc;
  olib::argv = (char **)argv;

  Cmdspec *cmdspec = getDepCmdspec();
  AutoCmdLine cmdline(cmdspec);

  if (getenv("DEPOPTS"))
    cmdline.processCmdline(FRAGMENTTYPE_OPTION, getenv("DEPOPTS"), DIVIDER_WHITESPACE);
  cmdline.processArgv(FRAGMENTTYPE_COMMAND, argc, argv);
  if (cmdline.haltline) {
    if (*argv[cmdline.paramIdx] == '-' || *argv[cmdline.paramIdx] == '/') {
      std::cerr << "dep: Unrecognized option " << argv[cmdline.paramIdx] << std::endl
        << "Exiting" << std::endl;
      return 1;
    }
  }
  if (cmdline.optionMap.count("-h")) {
    std::cout << "Usage: dep " << cmdspec->spec << std::endl;
    return 0;
  }
  if (cmdline.optionMap.count("-V")) {
    std::cout << "dep version 0.6" << std::endl;
    return 0;
  }
  if (!cmdline.optionMap.count("-f"))
    cmdline.optionMap["-f"].push_back("DepFile");

  RuleSet ruleset;
  ruleset.cmdline = &cmdline;

  File out("stdout", 1, std::cout);
  ruleset.readCache = 1;
  ruleset.writeCache = 1;
  ruleset.outputFile = &out;

  ruleset.processArgs(cmdline);

  std::string filename = cmdline.optionMap["-f"].back().c_str();

  DepTree cacheTree;
  RuleSet cacheset;
  cacheset.cmdline = &cmdline;
  cacheset.setVerbosity(-1);
  cacheset.tracefileSet = ruleset.tracefileSet;
  std::string cachefilename(joinFolderAndFile(DirName(filename.c_str()),
    std::string(".") + BaseName(filename.c_str()) + "Cache"));

  Timer timer;

  // Read the cache first. It is faster and easier to read, and has some information that the
  // main tree can use.

  File outputFile("stdout", 1, std::cout);

  if (ruleset.readCache) {
    cacheset.outputFile = &out;
    cacheTree.ruleset = &cacheset;
    if (cacheTree.readFile(cachefilename)) {
      std::cerr << "(dep: Can't read cache file " << cachefilename << ")\n";
      if (ruleset.readCache == 2) {
        std::cerr << "Quitting" << std::endl;
        return -1;
      }
      ruleset.readCache = 0;
    }
    if (ruleset.getVerbosity(VERBOSITY_TIMING, (Rule *)0))
      *outputFile.getOStream() << timer.getMicroseconds(TIMER_RESET)
        << " micros reading the caches, " << cacheTree.lexemes << " lexemes\n";
  }

  RuleSet *finalRuleset;
  int readDepFile;
  if (ruleset.readCache) {
    // This gets set properly by checking the DepFile times in the cache

    readDepFile = getenvint("READDEPFILE", 1);
  }
  else
    readDepFile = 1;

  DepTree zdepTree;
  if (readDepFile) {
    zdepTree.ruleset = &ruleset;
    ruleset.cacheset = ruleset.readCache? &cacheset: 0;
    int result = zdepTree.readFile(filename);
    if (ruleset.getVerbosity(VERBOSITY_TIMING, (Rule *)0))
      *outputFile.getOStream() << timer.getMicroseconds(TIMER_RESET)
        << " micros reading the DepFiles, " << zdepTree.lexemes << " lexemes\n";
    if (result != 0) {
      std::cerr << "dep: Can't read " << cmdline.optionMap["-f"].back() << std::endl;
      return 1;
    }
    finalRuleset = &ruleset;
  }
  else {
    finalRuleset = &cacheset;
    for (int i = 0; i < VERBOSITY_ARRAY_SIZE; i++)
      cacheset.verbosityArray[i] = ruleset.verbosityArray[i];
  }

  int result = finalRuleset->processCmd(argc - cmdline.paramIdx,
    argv + cmdline.paramIdx, &outputFile, /*&zdepTree,*/ timer);

  if (finalRuleset->getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile.getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros processing the command" << std::endl;
  if (ruleset.writeCache && finalRuleset->startedSomething || ruleset.writeCache == 2) {
    //logMsg << logValue(writeCache) << ' ' << logValue(finalRuleset->startedSomething) << std::endl;
    if (readDepFile)
      finalRuleset->combineWithCache(cacheset);
    copyFile(cachefilename, cachefilename + ".bak");
    std::ofstream ofs(cachefilename, std::ios::out | std::ios::trunc);
    if (!ofs.good()) {
      std::cerr << "dep: Can't open " << logValue(cachefilename) << " for output\n";
    }
    //std::cerr << "Outputting to " << logValue(cachefilename) << std::endl;
    finalRuleset->outputTo(ofs, 0);
  }
  if (finalRuleset->getVerbosity(VERBOSITY_TIMING, (Rule *)0))
    *outputFile.getOStream() << timer.getMicroseconds(TIMER_RESET)
      << " micros writing the cache" << std::endl;

  if (result == -1)
    return 1;
  if (WIFEXITED(result)) {
    if (WEXITSTATUS(result)) {
      std::cerr << "dep: child process exited with value " << WEXITSTATUS(result) << std::endl;
    }
    if (finalRuleset->failed)
      return 1;
    //return WEXITSTATUS(result);
  }
  if (WIFSIGNALED(result)) {
    std::cerr << "dep: child process terminated by signal "
      << WTERMSIG(result) << ' ' << SignalToMacroName(WTERMSIG(result)) << std::endl;
    return 13;
  }
  if (result == 65535) {
    std::cerr << "dep: couldn't start build command" << std::endl;
    return 133;
  }
  if (!WIFEXITED(result)) {
    logMsg << "Process terminated by something other than exit or signal "
      << logValue(result) << std::endl;
    return 113;
  }
  return 0;
}

