
// logMsg.cpp
// Copyright 2015 Matthew Rickard
// This file is part of dep

// dep is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#include "precomp.h"

#include "logMsg.h"
#include "stringify.h"

#include <string.h>
#include <vector>
#include <fstream>

int logMsgIndent = 0;

LogMsgIndenter::LogMsgIndenter(): closed(0) {
  logMsgIndent++;
}

LogMsgIndenter::~LogMsgIndenter() {
  close();
}

void LogMsgIndenter::close() {
  if (!closed) {
    closed = 1;
    logMsgIndent--;
  }
}

// New improved logMsg responds to LOGMSG environment variable
// LOGMSG=indent,file,line,fn shows all of those
// LOGMSG=noprefix turns off all prefixes
// LOGMSG=noindent turns off indentation
// LOGMSG=none turns off all log messaging

#define LOGMSG_INDENT 1
#define LOGMSG_FILE   2
#define LOGMSG_LINE   3
#define LOGMSG_FN     4

int startsWith(const char **env, const char *val) {
  int len = strlen(val);
  int result = !strncmp(*env, val, len);
  if (result)
    *env += len;
  //std::cerr << "Does " << *env << " start with " << val << "? " << result << std::endl;
  return result;
}

std::string logLine_fn(const char *file, int line, const char *fn) {
  static int initialized = 0;
  static std::vector<int> v;
  if (!initialized) {
    const char *env = getenv("LOGMSG");
    int indent = 1;
    int prefix = 1;
    if (env)
      for (;;) {
        //std::cerr << logValue(env) << std::endl;
        if (startsWith(&env, "indent"))
          v.push_back(LOGMSG_INDENT);
        else if (startsWith(&env, "file"))
          v.push_back(LOGMSG_FILE);
        else if (startsWith(&env, "line"))
          v.push_back(LOGMSG_LINE);
        else if (startsWith(&env, "fn"))
          v.push_back(LOGMSG_FN);
        else if (startsWith(&env, "noindent"))
          indent = 0;
        else if (startsWith(&env, "noprefix"))
          prefix = 0;
        else if (!*env)
          break;
        else if (startsWith(&env, ","))
          ;
        else
          break;
      }
    if (prefix && !v.size()) {
      if (indent)
        v.push_back(LOGMSG_INDENT);
      v.push_back(LOGMSG_FILE);
      v.push_back(LOGMSG_LINE);
      v.push_back(LOGMSG_FN);
    }
    initialized = 1;
  }
  std::string result;
  for (auto i: v) {
    if (i == LOGMSG_INDENT)
      result += std::string(2 * logMsgIndent, ' ');
    else {
      if (i == LOGMSG_FN) {
        if (fn && *fn)
          result += fn;
        else
          continue;
      }
      else if (i == LOGMSG_FILE)
        result += BaseName(file);
      else if (i == LOGMSG_LINE)
        result += stringify(line);
      result.push_back(' ');
    }
  }
  return result;
}

std::ostream *logStream() {
  static int initialized = 0;
  static int enabled = 1;
  if (!initialized) {
    static const char *env = getenv("LOGMSG");
    if (env)
      if (startsWith(&env, "none"))
        enabled = 0;
    initialized = 1;
  }
  if (enabled)
    return &std::cerr;
  static std::ofstream *devNullStream = new std::ofstream("/dev/null");
  return devNullStream;
}

